#include "mainwindow.h"
#include <QtGui/QMessageBox>
#include <QtGui/QFileDialog>

#include "../../CrossPlatform/DemoDBGenerator.h"
#include "../../CrossPlatform/MetaDB.h"

#ifdef DB_DEPLOY
  #include "../3rdParty/quazip/quazip.h"
  #include "../3rdParty/quazip/quazipfile.h"
#endif

MainWindow::MainWindow(QWidget *parent)
    : QWidget(parent),
      m_BuildingImagePath(""),
      m_strCurrentFile(""),
      m_pConfDB(new ConferenceDB()),
      m_bEditBlockSignals(false)
{
    setupUi();
}

MainWindow::~MainWindow()
{
    delete m_pConfDB;
}

void MainWindow::SetTitle() {
    QFileInfo fi(m_strCurrentFile);

    if (m_strCurrentFile.isEmpty())
        this->setWindowTitle(tr("Conference Manager DB Editor %1 [*]").arg(VERSION));
    else
        this->setWindowTitle(tr("Conference Manager DB Editor %1 - %2 [*]").arg(VERSION).arg(fi.fileName()));
}

void MainWindow::setupUi() {
    this->resize(900, 600);
    m_globalLayout = new QVBoxLayout(this);

    m_globalTabs = new QTabWidget(this);
    setupDataTabs();

    m_New = new QPushButton(this);
    m_New->setText("New");
    m_Cleanup = new QPushButton(this);
    m_Cleanup->setText("Cleanup");
    m_Demo = new QPushButton(this);
    m_Demo->setText("Generate Demo");
    m_Open = new QPushButton(this);
    m_Open->setText("Open");
    m_Save = new QPushButton(this);
    m_Save->setText("Save");
    m_SaveAs = new QPushButton(this);
    m_SaveAs->setText("Save as ...");
#ifdef DB_DEPLOY
    m_Deploy = new QPushButton(this);
    m_Deploy->setText("Deploy");
#endif

#ifdef DB_SPECIAL
    m_Special = new QPushButton(this);
    m_Special->setText("Special Import");
#endif

    m_Quit = new QPushButton(this);
    m_Quit->setText("Quit");

    m_ButtonLayout = new QHBoxLayout();
    m_ButtonLayout->setObjectName("ButtonLayout");
    m_ButtonLayout->addWidget(m_New);
    m_ButtonLayout->addWidget(m_Cleanup);
    m_ButtonLayout->addWidget(m_Demo);
    m_ButtonLayout->addWidget(m_Open);
    m_ButtonLayout->addWidget(m_Save);
    m_ButtonLayout->addWidget(m_SaveAs);
#ifdef DB_DEPLOY
    m_ButtonLayout->addWidget(m_Deploy);
    QObject::connect(m_Deploy, SIGNAL(clicked()), this, SLOT(deployDB()));
#endif
#ifdef DB_SPECIAL
    m_ButtonLayout->addWidget(m_Special);
    QObject::connect(m_Special, SIGNAL(clicked()), this, SLOT(specialImportDB()));
#endif
    m_ButtonLayout->addWidget(m_Quit);

    m_globalLayout->setObjectName("GlobalLayout");
    m_globalLayout->addWidget(m_globalTabs);
    m_globalLayout->addLayout(m_ButtonLayout);

    QObject::connect(m_New, SIGNAL(clicked()), this, SLOT(newDB()));
    QObject::connect(m_Cleanup, SIGNAL(clicked()), this, SLOT(cleanup()));
    QObject::connect(m_Open, SIGNAL(clicked()), this, SLOT(openDB()));
    QObject::connect(m_Demo, SIGNAL(clicked()), this, SLOT(genDemoDB()));
    QObject::connect(m_Save, SIGNAL(clicked()), this, SLOT(saveDB()));
    QObject::connect(m_SaveAs, SIGNAL(clicked()), this, SLOT(saveDBAs()));
    QObject::connect(m_Quit, SIGNAL(clicked()), this, SLOT(quit()));
}

void MainWindow::setupDataTabs() {
    SetTitle();
    setupGeneralTab();
    setupAffiliationsTab();
    setupPeopleTab();
    setupPublicationsTab();
    setupTalksTab();
    setupBuildingsTab();
    setupRoomsTab();
    setupSessionsTab();
    setupTracksTab();
    setupDaysTab();
}

bool MainWindow::SaveCheck(QString reason) {
    if (m_pConfDB && this->isWindowModified()) {
        QMessageBox msgBox;
        msgBox.setText("The database has been modified.");
        msgBox.setInformativeText(tr("Do you want to save your changes before %1?").arg(reason));
        msgBox.setStandardButtons(QMessageBox::Save | QMessageBox::Discard | QMessageBox::Cancel);
        msgBox.setDefaultButton(QMessageBox::Save);
        int ret = msgBox.exec();

        switch (ret) {
            case QMessageBox::Save:
                return saveDB();
            case QMessageBox::Discard:
                return true;
            case QMessageBox::Cancel:
                return false;
        }
        return false;
    }
    return true;
}

void MainWindow::UpdateUI() {
    SetTitle();
    UpdateGeneralUI();
    UpdateAffUI();
    UpdatePeopleUI();
    UpdatePublicationUI();
    UpdateTalksUI();
    UpdateBuildingsUI();
    UpdateRoomsUI();
    UpdateSessionsUI();
    UpdateTracksUI();
    UpdateDaysUI();
}

// Slots

void MainWindow::conferenceDataChanged() {
    if (m_bEditBlockSignals) return;
    SetModifyState(true);
}

void MainWindow::SetModifyState(bool bModified) {
   this->setWindowModified(bModified);
}

// Buttons

void MainWindow::newDB() {
    if (!SaveCheck("starting a new database")) return;
    delete m_pConfDB;
    m_pConfDB = new ConferenceDB();
    SetModifyState(false);
    m_strCurrentFile = "";
    UpdateUI();
}


void MainWindow::cleanup() {
    unsigned int iChanged = 0;
    unsigned int iIterationChanged = 0;

    // deleteing an unrefernced entry may create new unreferenced
    // entries
    do {
      iIterationChanged = m_pConfDB->RemoveUnusedEntries();
      iChanged += iIterationChanged;
    } while (iIterationChanged);

    QMessageBox msgBox;
    msgBox.setIcon(QMessageBox::Information);
    if (iChanged > 0) {
        SetModifyState(true);
        UpdateUI();

        if (iChanged > 1) {
            msgBox.setText(tr("%1 unreferenced entries were removed from the database.").arg(iChanged));
        } else {
            msgBox.setText("One unreferenced entry was removed from the database.");
        }
    } else {
        msgBox.setText("No unreferenced entries found.");
    }
    msgBox.exec();
}

void MainWindow::genDemoDB() {
    if (!SaveCheck("starting a new database")) return;
    delete m_pConfDB;
    m_pConfDB = DemoDBGenerator::GenDemoDB();
    SetModifyState(true);
    m_strCurrentFile = "";
    UpdateUI();
}

void MainWindow::openDB() {
    if (!SaveCheck("opening another database")) return;

    QString path = QFileDialog::getOpenFileName(this,"Choose a conference database", QString::null, "Conference Databases (*.db *.txt)");

    if (path.count() > 0) {
        ConferenceDB* tmp = new ConferenceDB(std::string(path.toUtf8()));

        if (tmp->DidConstructionFail()) {
            QMessageBox msgBox;
            msgBox.setIcon(QMessageBox::Critical);
            msgBox.setText("The database failed to load!");
            msgBox.exec();
            return;
        }

        delete m_pConfDB;
        m_pConfDB = tmp;        
        SetModifyState(false);
        m_strCurrentFile = path;        
        UpdateUI();
    }
}

bool MainWindow::saveDB(QString path) {
    if (!m_pConfDB) return false;

    if (path.count() > 0) {
        bool bResult = m_pConfDB->SerializeFile(std::string(path.toUtf8()));
        if (bResult) SetModifyState(false);

        if (!bResult) {
            QMessageBox msgBox;
            msgBox.setIcon(QMessageBox::Critical);
            msgBox.setText("The database failed to save!");
            msgBox.exec();
        } else {
            m_strCurrentFile = path;
            SetTitle();
        }
        return bResult;
    }
    return false;
}

bool MainWindow::saveDB() {
  if (m_strCurrentFile.isEmpty())
      return saveDBAs();
  else
      return saveDB(m_strCurrentFile);
}

bool MainWindow::saveDBAs() {
    QString path = QFileDialog::getSaveFileName(this,"Store to a conference database", QString::null, "Conference Databases (*.db *.txt)");
    return saveDB(path);
}

#ifdef DB_DEPLOY

bool MainWindow::Compress(const std::vector < std::pair<QFileInfo, QString> >& files, const QString& target)
{
    QFile zipFile(target);
    QuaZip zip(&zipFile);
    if(!zip.open(QuaZip::mdCreate)) {
        return false;
    }
    zip.setComment("ConfManDatabase");


    QFile inFile;
    QuaZipFile outFile(&zip);
    char c;
    for (size_t i = 0;i<files.size();++i) {
        QFileInfo file = files[i].first;

        inFile.setFileName(file.filePath());
        if(!inFile.open(QIODevice::ReadOnly)) {
            return false;
        }
        if(!outFile.open(QIODevice::WriteOnly, QuaZipNewInfo(files[i].second, file.filePath() ))) {
            return false;
        }
        qint64 pos = 0;
        while (inFile.getChar(&c)&&outFile.putChar(c)) {
            char buf[4096];
            qint64 l = inFile.read(buf, 4096);
            if (l < 0) {
                break;
            }
            if (l == 0)
                break;
            if (outFile.write(buf, l) != l) {
                break;
            }
            pos += l;
        }
        if(outFile.getZipError()!=UNZ_OK) {
            return false;
        }
        outFile.close();
        if(outFile.getZipError()!=UNZ_OK) {
            return false;
        }
        inFile.close();
    }
    zip.close();
    if(zip.getZipError()!=0) {
        return false;
    }
    return true;
}

void MainWindow::deployDB() {
    if (!SaveCheck("deploying")) return;

    QFileInfo dbFile(m_strCurrentFile);

    QString targetDir = QFileDialog::getExistingDirectory(this, "Specify Server Deployment Directory");
    if (targetDir.isEmpty()) return;

    QString targetFileName = tr("%1.zip").arg(dbFile.baseName());
    QString targetFilePath = tr("%1/%2").arg(targetDir).arg(targetFileName);

    std::vector < std::pair<QFileInfo, QString> > compressList;

    QFileInfo possibleImageFilename(tr("%1/logo.png").arg(dbFile.dir().canonicalPath()));
    if (possibleImageFilename.exists()) {
        compressList.push_back(std::make_pair(possibleImageFilename.filePath(), tr("logo.png")));
    } else {

        QMessageBox::StandardButton reply = QMessageBox::question(this, tr("Conference Logo"),
                                             tr("You you want to add a conference logo?"),
                                             QMessageBox::Yes | QMessageBox::No | QMessageBox::Cancel, QMessageBox::Yes);
        if (reply == QMessageBox::Yes) {
            QString imageFilename = QFileDialog::getOpenFileName(this,"Open conference logo", QString::null, "PNG Images (*.png)");
            if (imageFilename.isEmpty()) return;
            QFileInfo imageFile(imageFilename);
            compressList.push_back(std::make_pair(imageFile, tr("logo.png")));

            reply = QMessageBox::question(this, tr("Save Logo"),
                                                 tr("You you want to make a copy of this logo file so it will be automatically used in the future?"),
                                                 QMessageBox::Yes | QMessageBox::No, QMessageBox::Yes);

            if (reply == QMessageBox::Yes)
                QFile::copy(imageFile.filePath(),  possibleImageFilename.filePath());

        } else if (reply == QMessageBox::Cancel) {
            return;
        }
    }

    compressList.push_back(std::make_pair(dbFile, tr("Conference.db")));
    std::vector<std::string> files = m_pConfDB->GetSupportFileList();
    for (size_t i = 0;i<files.size();i++) {
        QFileInfo supportFile(tr("%1/%2").arg(dbFile.dir().canonicalPath()).arg(files[i].c_str()));
        if (supportFile.exists())
            compressList.push_back(std::make_pair(supportFile, supportFile.fileName()));
    }

    if (!Compress(compressList, targetFilePath)) {
        QMessageBox msgBox;
        msgBox.setIcon(QMessageBox::Critical);
        msgBox.setText(tr("Unable to deploy the database!"));
        msgBox.exec();
        return;
    }


    // check if we have a metadata db in place already

    std::string metaDBFilename(tr("%1/Meta.db").arg(targetDir).toUtf8());

    ServerMetaDB metaDB(metaDBFilename);

    if (metaDB.DidConstructionFail()) {
        ServerMetaDB newMetaDB;

        ServerConferenceFileMD conferenceFile(m_pConfDB->GetTitle(),
                                              m_pConfDB->GetShortTitle(),
                                              0,
                                              std::string(targetFileName.toUtf8()),
                                              MD5::ComputeMD5(std::string(targetFilePath.toUtf8())));

        newMetaDB.m_ConfFiles.push_back(conferenceFile);

        if (newMetaDB.SerializeFile(metaDBFilename)) {
            QMessageBox msgBox;
            msgBox.setIcon(QMessageBox::Information);
            msgBox.setText(tr("The conference has been deployed to a new database server!"));
            msgBox.exec();
            return;
        } else {
            QMessageBox msgBox;
            msgBox.setIcon(QMessageBox::Critical);
            msgBox.setText(tr("Unable to save server metadata!"));
            msgBox.exec();
            return;
        }
    } else {
        // try to find the db in the metadata

        int iVersion = 0;
        size_t index = metaDB.m_ConfFiles.size();
        for (size_t i = 0;i<metaDB.m_ConfFiles.size();++i) {
            if (metaDB.m_ConfFiles[i].m_strTitle ==  m_pConfDB->GetTitle()) {
                index = i;
                break;
            }
        }

        if (index < metaDB.m_ConfFiles.size()) {
            metaDB.m_ConfFiles[index].m_iVersion++;
            iVersion = metaDB.m_ConfFiles[index].m_iVersion;
            metaDB.m_ConfFiles[index].m_vMD5Hash = MD5::ComputeMD5(std::string(targetFilePath.toUtf8()));
        } else {
            ServerConferenceFileMD conferenceFile(m_pConfDB->GetTitle(),
                                                  m_pConfDB->GetShortTitle(),
                                                  0,
                                                  std::string(targetFileName.toUtf8()),
                                                  MD5::ComputeMD5(std::string(targetFilePath.toUtf8())));

            metaDB.m_ConfFiles.push_back(conferenceFile);
        }

        if (metaDB.SerializeFile(metaDBFilename)) {
            QMessageBox msgBox;
            msgBox.setIcon(QMessageBox::Information);
            if ( iVersion > 0)
                msgBox.setText(tr("The conference has been updated to version %1 on the database server!").arg(iVersion));
            else
                msgBox.setText(tr("The conference has been added as a new database to the existing database server!"));
            msgBox.exec();
            return;
        } else {
            QMessageBox msgBox;
            msgBox.setIcon(QMessageBox::Critical);
            msgBox.setText(tr("Unable to save server metadata!"));
            msgBox.exec();
            return;
        }

    }

}

#endif

#ifdef DB_SPECIAL

#include "../../CrossPlatform/Paper.h"
#include "../../CrossPlatform/Poster.h"
#include "../../CrossPlatform/Affiliation.h"
#include "../../CrossPlatform/SciPerson.h"

std::vector<int> GetIDs(const std::vector< std::string >& records, size_t iStartIndex ) {
    std::vector<int> intElements;
    for (size_t i = iStartIndex;i<records.size();++i) {
        if (!records[i].empty())
            intElements.push_back(atoi(records[i].c_str()));
    }
    return intElements;
}

void MainWindow::specialImportDB() {
/*
    while (m_pConfDB->people.Count()) {
        m_pConfDB->people.RemoveEntryByPos(0);
    }

    while (m_pConfDB->affiliations.Count()) {
        m_pConfDB->affiliations.RemoveEntryByPos(0);
    }
*/

    QString authorFile = QFileDialog::getOpenFileName(this,"Choose a author csv", QString::null, "csv (*.csv *.txt)");
    QString paperFile = QFileDialog::getOpenFileName(this,"Choose a paper csv", QString::null, "csv (*.csv *.txt)");
    QString posterFile = QFileDialog::getOpenFileName(this,"Choose a poster csv", QString::null, "csv (*.csv *.txt)");

    // first read author file
    CSFParser authorParser(std::string(authorFile.toUtf8()),0, true);
    authorParser.SetRecordSizeToMax();
    std::map< int, index_t > mapping;
    for (size_t i = 0;i<authorParser.RecordCount();i++) {
        std::vector< std::string > record = authorParser.GetRecord(i);
        int iCSV_ID = atoi(record[0].c_str());
        SciPerson author(record[1], record[2], record[3], "","","","");

        for (size_t j = 4;j<record.size();j++) {
            if (!record[j].empty()) {
                Affiliation aff(record[j], "");
                index_t dbAffID = m_pConfDB->affiliations.AddUniqueEntry(aff);
                author.AddAffiliation(*m_pConfDB->affiliations.GetEntryByIndex(dbAffID));
            }
        }

        index_t dbID = m_pConfDB->people.AddUniqueEntry(author);
        mapping[iCSV_ID] = dbID;
    }
    // next, the paper file
    CSFParser paperParser(std::string(paperFile.toUtf8()),0, true);
    paperParser.SetVariableRecordSize();
    for (size_t i = 0;i<paperParser.RecordCount();i++) {
        std::vector< std::string > record = paperParser.GetRecord(i);
        Paper paper(record[0], record[1], "", "","");
        std::vector<int> authorTableIDs = GetIDs(record,2);
        for (size_t j = 0;j<authorTableIDs.size()-1;++j) {
            const SciPerson* author = m_pConfDB->people.GetEntryByIndex(mapping[authorTableIDs[j]]);
            if (author) paper.AddAuthor(*author);
        }
        m_pConfDB->works.AddUniqueEntry(paper);
    }

    // now, the poster file
    CSFParser posterParser(std::string(posterFile.toUtf8()),0, true);
    posterParser.SetVariableRecordSize();
    for (size_t i = 0;i<posterParser.RecordCount();i++) {
        std::vector< std::string > record = posterParser.GetRecord(i);
        Poster poster(record[0], record[1], "", "","");
        std::vector<int> authorTableIDs = GetIDs(record,2);
        for (size_t j = 0;j<authorTableIDs.size()-1;++j) {
            const SciPerson* author = m_pConfDB->people.GetEntryByIndex(mapping[authorTableIDs[j]]);
            if (author) poster.AddAuthor(*author);
        }
        m_pConfDB->works.AddUniqueEntry(poster);
    }

    SetModifyState(true);
    UpdateUI();
}

#endif



void MainWindow::quit() {
    close();
}

void MainWindow::closeEvent(QCloseEvent *event)
{
    if (SaveCheck("closing the application")) {
        event->accept();
    } else {
        event->ignore();
    }
}


bool MainWindow::StartEditBlocker() {
    bool bTemp = m_bEditBlockSignals;
    m_bEditBlockSignals = true;
    return bTemp;
}

void MainWindow::EndEditBlocker(bool bTemp) {
    m_bEditBlockSignals = bTemp;
}
