#include "mainwindow.h"

#include <QtGui/QMessageBox>
#include <QtGui/QInputDialog>
#include "QImportDialog.h"
#include <algorithm>

void MainWindow::setupTracksTab(){
    m_tabTracks = new QWidget();
    m_tabTracks->setObjectName(QString::fromUtf8("m_tabTracks"));
    m_tabTracks->setWindowTitle(QString::fromUtf8("Tracks"));
    m_globalTabs->addTab(m_tabTracks, m_tabTracks->windowTitle());

    m_TrackGlobalLayout = new QHBoxLayout(m_tabTracks);
    m_TrackGlobalLayout->setSpacing(6);
    m_TrackGlobalLayout->setContentsMargins(0, 0, 0, 0);
    m_TrackGlobalLayout->setObjectName(QString::fromUtf8("m_TrackGlobalLayout"));
    m_TrackSplitter = new QSplitter();
    m_TrackGlobalLayout->addWidget(m_TrackSplitter);

    m_TrackListFrame = new QFrame(m_tabTracks);
    m_TrackListFrame->setObjectName(QString::fromUtf8("m_TrackListFrame"));
    m_TrackListFrame->setFrameShape(QFrame::NoFrame);
    m_TrackListFrame->setLineWidth(0);
    m_gridLayoutTrackList = new QGridLayout(m_TrackListFrame);
    m_gridLayoutTrackList->setSpacing(6);
    m_gridLayoutTrackList->setContentsMargins(0, 0, 0, 0);
    m_gridLayoutTrackList->setObjectName(QString::fromUtf8("m_gridLayoutTrackList"));

    m_TrackImportBtn = new QPushButton(m_tabTracks);
    m_TrackImportBtn->setText("Import Tracks from CSV-File");
    m_gridLayoutTrackList->addWidget(m_TrackImportBtn, 0, 0, 1, 1);

    m_TrackList = new QElemList(m_tabTracks,true);
    m_TrackList->setObjectName(QString::fromUtf8("m_TrackList"));
    m_gridLayoutTrackList->addWidget(m_TrackList, 1, 0, 1, 1);

    m_TrackEditFrame = new QFrame(m_tabTracks);
    m_TrackEditFrame->setObjectName(QString::fromUtf8("m_TrackEditFrame"));
    m_TrackEditFrame->setFrameShape(QFrame::NoFrame);
    m_TrackEditFrame->setFrameShadow(QFrame::Raised);
    m_TrackEditFrame->setLineWidth(0);
    m_gridLayoutTrack = new QGridLayout(m_TrackEditFrame);
    m_gridLayoutTrack->setSpacing(6);
    m_gridLayoutTrack->setContentsMargins(0, 0, 0, 0);
    m_gridLayoutTrack->setObjectName(QString::fromUtf8("m_gridLayoutTrack"));

    m_TrackNameLabel = new QLabel(m_TrackEditFrame);
    m_TrackNameLabel->setObjectName(QString::fromUtf8("m_TrackNameLabel"));
    m_TrackNameLabel->setText(QString::fromUtf8("Description"));
    m_gridLayoutTrack->addWidget(m_TrackNameLabel, 0, 0, 1, 1);
    m_TrackNameEdit = new QLineEdit(m_TrackEditFrame);
    m_TrackNameEdit->setObjectName(QString::fromUtf8("m_TrackNameEdit"));
    m_gridLayoutTrack->addWidget(m_TrackNameEdit, 0, 1, 1, 1);

    m_TrackSessionLabel = new QLabel(m_TrackEditFrame);
    m_TrackSessionLabel->setObjectName(QString::fromUtf8("m_TrackSessionLabel"));
    m_TrackSessionLabel->setText(QString::fromUtf8("Sessions"));
    m_gridLayoutTrack->addWidget(m_TrackSessionLabel, 7, 0, 1, 2);

    m_TrackSessionList = new QElemList(m_TrackEditFrame, false, false);
    m_TrackSessionList->setObjectName(QString::fromUtf8("m_TrackSessionList"));
    m_gridLayoutTrack->addWidget(m_TrackSessionList, 8, 0, 1, 2);

    m_TrackEditSpacer = new QSpacerItem(20, 40, QSizePolicy::Minimum, QSizePolicy::Expanding);
    m_gridLayoutTrack->addItem(m_TrackEditSpacer, 9, 0, 1, 1);

    m_TrackSplitter->addWidget(m_TrackListFrame);
    m_TrackSplitter->addWidget(m_TrackEditFrame);

    QObject::connect(m_TrackNameEdit, SIGNAL(textEdited(QString)), this, SLOT(tracksDataChanged()));
    QObject::connect(m_TrackNameEdit, SIGNAL(textEdited(QString)), this, SLOT(conferenceDataChanged()));

    QObject::connect(m_TrackList, SIGNAL(newElement()), this, SLOT(tracksNewElement()));
    QObject::connect(m_TrackList, SIGNAL(newElement()), this, SLOT(conferenceDataChanged()));
    QObject::connect(m_TrackList, SIGNAL(delElement(index_t)), this, SLOT(tracksDelElement(index_t)));
    QObject::connect(m_TrackList, SIGNAL(delElement(index_t)), this, SLOT(conferenceDataChanged()));
    QObject::connect(m_TrackList, SIGNAL(activeElement(index_t)), this, SLOT(tracksActiveElement(index_t)));

    QObject::connect(m_TrackSessionList, SIGNAL(newElement()), this, SLOT(tracksSessionNewElement()));
    QObject::connect(m_TrackSessionList, SIGNAL(delElement(index_t)), this, SLOT(tracksSessionDelElement(index_t)));
    QObject::connect(m_TrackSessionList, SIGNAL(delElement(index_t)), this, SLOT(conferenceDataChanged()));

    QObject::connect(m_TrackImportBtn, SIGNAL(clicked()), this, SLOT(tracksImportCSV()));


    tracksActiveElement(INDEX_EMPTY);
}

void MainWindow::tracksImportCSV() {
    if (!SaveCheck("importing tracks from CSV file")) return;

    std::vector<std::string> properties;
    properties.push_back("Name");

    QImportDialog d(properties, this);
    if (QDialog::Accepted != d.exec()) return;

    std::string filename = d.GetFilename();
    if (!filename.empty()) {
        std::vector<int> choices = d.GetChoices();
        CSFParser csvData(filename,d.GetOffset(),d.RemoveWhitespaces());

        csvData.SetRecordSizeToMax();

        for (size_t i = 0;i<csvData.RecordCount();i++) {
            std::vector< std::string > csvRecord = csvData.GetRecord(i);
            Track track(choices[0]>= 0 ? csvRecord[choices[0]] : "");
            m_pConfDB->tracks.AddUniqueEntry(track);
        }

        conferenceDataChanged();
        UpdateUI();
    }
}

void MainWindow::UpdateTracksUI(index_t newIndex) {
    std::vector< ElemListEntry > trackList;
    for (size_t i = 0;i<m_pConfDB->tracks.Count();++i) {
        const Track* t  = m_pConfDB->tracks.GetEntryByPos(i);

        std::stringstream ss;
        if (!t->GetTitle().empty())
            ss << t->GetTitle().c_str() << ": ";

        if (t->GetSessions().size()) {
            int start = t->GetStartTime(m_pConfDB);
            int end = t->GetEndTime(m_pConfDB);

            QTime startTime(start/60, start%60);
            QTime endTime(end/60, end%60);

            ss << std::string(startTime.toString("hh:mm ap").toUtf8()).c_str() << " - " << std::string(endTime.toString("hh:mm ap").toUtf8()).c_str();
        } else {
            ss << "no sessions in this track";
        }

        trackList.push_back( ElemListEntry(ss.str(),
                                           t->GetIndex(),
                                           t->GetReferenceCount(m_pConfDB))  );
    }
    m_TrackList->ElemUpdate(trackList, newIndex);

    UpdateTrackSessionList();
}

void MainWindow::UpdateTrackSessionList() {
    UpdateTrackSessionList(m_TrackList->GetActiveIndex());
}

static bool sortSecond (const std::pair<index_t, int>& i, const std::pair<index_t, int>& j) {
    return (i.second<j.second);
}

void MainWindow::UpdateTrackSessionList(index_t index) {
    const Track* t = m_pConfDB->tracks.GetEntryByIndex(index);

    std::vector< ElemListEntry > sessList;
    if (t) {
        std::vector< std::pair<index_t, int> > elems;
        for (size_t i = 0;i<t->GetSessions().size();++i) {
            index_t sessIndex = t->GetSessions()[i];
            const Session* session = m_pConfDB->sessions.GetEntryByIndex(sessIndex);
            elems.push_back(std::make_pair(session->GetIndex(), session->GetStartTime()));
        }
        // sort by start time
        std::stable_sort(elems.begin(), elems.end(), sortSecond);
        for (size_t i = 0;i<elems.size();++i) {
            index_t sessIndex = elems[i].first;
            const Session* session = m_pConfDB->sessions.GetEntryByIndex(sessIndex);
            sessList.push_back( ElemListEntry(elemToString(session,false).c_str(),
                                              session->GetIndex()));
        }
    }
    m_TrackSessionList->ElemUpdate(sessList, INDEX_EMPTY);
}

// Slots

void MainWindow::tracksDataChanged() {
    index_t index = m_TrackList->GetActiveIndex();
    Track* t = m_pConfDB->tracks.GetEntryByIndex(index);

    t->SetTitle(std::string(m_TrackNameEdit->text().toUtf8()));

    UpdateTracksUI(index);
    UpdateDayTrackList();
}

void MainWindow::tracksNewElement() {
    Track newTrack("New");
    index_t index = m_pConfDB->tracks.AddEntry(newTrack);
    UpdateTracksUI(index);
    UpdateDayTrackList();
}

void MainWindow::tracksDelElement(index_t index) {
    Track* t = m_pConfDB->tracks.GetEntryByIndex(index);
    assert(t);
    assert(t->GetReferenceCount(m_pConfDB) == 0);

    m_pConfDB->tracks.RemoveEntryByIndex(index);
    UpdateTracksUI(INDEX_EMPTY);
    UpdateSessionsUI();
    UpdateDayTrackList();
}

void MainWindow::tracksActiveElement(index_t index) {
    const Track* t = m_pConfDB->tracks.GetEntryByIndex(index);
    if (t) {
        m_TrackNameEdit->setText(QString::fromUtf8(t->GetTitle().c_str()));
        UpdateTrackSessionList(index);
        m_TrackNameEdit->setEnabled(true);
        m_TrackSessionList->setEnabled(true);
    } else {
        m_TrackNameEdit->setText("");
        m_TrackSessionList->Clear();
        m_TrackNameEdit->setEnabled(false);
        m_TrackSessionList->setEnabled(false);
    }

}

std::string MainWindow::elemToString(const Session* s, bool bIncludeID) {
    std::stringstream ss;

    int startTime = s->GetStartTime();
    int endTime = s-> GetEndTime();

    ss << s->GetDescription().c_str() << ": " << s->GetTitle().c_str() << " " << std::string(QTime(startTime/60, startTime%60).toString("H:mm").toUtf8()).c_str()
                               << "-"
                               <<std::string(QTime(endTime/60, endTime%60).toString("H:mm").toUtf8()).c_str();

    if (bIncludeID)
        ss << " [ID=" << s->GetIndex() << "]";
    return ss.str();
}

void MainWindow::tracksSessionNewElement() {
    index_t tIndex = m_TrackList->GetActiveIndex();
    Track* t = m_pConfDB->tracks.GetEntryByIndex(tIndex);
    assert(t);

    QStringList items;
    for (size_t i = 0;i<m_pConfDB->sessions.Count();++i) {
        const Session* s = m_pConfDB->sessions.GetEntryByPos(i);
        if (!s) continue;

        // only display sessions that are not in use elsewhere
        if (s->GetReferenceCount(m_pConfDB) == 0) {
            items << elemToString(s, true).c_str();
        }
    }

    if (items.count() == 0) {
        QMessageBox msgBox;
        msgBox.setIcon(QMessageBox::Warning);
        msgBox.setText("No additional sessions available.");
        msgBox.exec();
        return;
    }

    items.sort();

    bool ok;
    QString item = QInputDialog::getItem(this, tr("Select an additional Session"),
                                         tr("Sessions:"), items, 0, false, &ok);
    if (ok && !item.isEmpty()) {


        for (size_t i = 0;i<m_pConfDB->sessions.Count();++i) {
            const Session* s = m_pConfDB->sessions.GetEntryByPos(i);
            if (item == QString::fromUtf8( elemToString(s, true).c_str() ) ) {
                t->AddSession(*s);
                break;
            }
        }

        UpdateTracksUI(tIndex);
        UpdateTrackSessionList();
        UpdateSessionsUI();
        UpdateDaysUI();
        conferenceDataChanged();
    }

}

void MainWindow::tracksSessionDelElement(index_t index) {
    index_t tIndex = m_TrackList->GetActiveIndex();
    Track* t = m_pConfDB->tracks.GetEntryByIndex(tIndex);

    t->DelSession(index);

    UpdateTracksUI(tIndex);
    UpdateSessionsUI();
    UpdateDaysUI();
}
