#include "mainwindow.h"

#include <QtGui/QMessageBox>
#include <QtGui/QInputDialog>
#include <algorithm>
#include "QImportDialog.h"

#include "../../CrossPlatform/InvitedTalk.h"
#include "../../CrossPlatform/PaperTalk.h"

void MainWindow::setupTalksTab(){
    m_tabTalks = new QWidget();
    m_tabTalks->setObjectName(QString::fromUtf8("m_tabTalks"));
    m_tabTalks->setWindowTitle(QString::fromUtf8("Talks"));
    m_globalTabs->addTab(m_tabTalks, m_tabTalks->windowTitle());

    m_TalkGlobalLayout = new QHBoxLayout(m_tabTalks);
    m_TalkGlobalLayout->setSpacing(6);
    m_TalkGlobalLayout->setContentsMargins(0, 0, 0, 0);
    m_TalkGlobalLayout->setObjectName(QString::fromUtf8("m_TalkGlobalLayout"));
    m_TalkSplitter = new QSplitter();
    m_TalkGlobalLayout->addWidget(m_TalkSplitter);

    m_TalkListFrame = new QFrame(m_tabTalks);
    m_TalkListFrame->setObjectName(QString::fromUtf8("m_TalkListFrame"));
    m_TalkListFrame->setFrameShape(QFrame::NoFrame);
    m_TalkListFrame->setLineWidth(0);
    m_gridLayoutTalkList = new QGridLayout(m_TalkListFrame);
    m_gridLayoutTalkList->setSpacing(6);
    m_gridLayoutTalkList->setContentsMargins(0, 0, 0, 0);
    m_gridLayoutTalkList->setObjectName(QString::fromUtf8("m_gridLayoutTalkList"));
    m_TalkImportBtn = new QPushButton(m_tabTalks);
    m_TalkImportBtn->setText("Import Talks from CSV-File");
    m_gridLayoutTalkList->addWidget(m_TalkImportBtn, 0, 0, 1, 1);
    m_TalkTransferBtn = new QPushButton(m_tabTalks);
    m_TalkTransferBtn->setText("Transfer all papers");
    m_gridLayoutTalkList->addWidget(m_TalkTransferBtn, 0, 1, 1, 1);

    m_TalkList = new QElemList(m_tabTalks, true);
    m_TalkList->setObjectName(QString::fromUtf8("m_TalkList"));
    m_gridLayoutTalkList->addWidget(m_TalkList, 1, 0, 1, 2);

    m_TalkEditFrame = new QFrame(m_tabTalks);
    m_TalkEditFrame->setObjectName(QString::fromUtf8("m_TalkEditFrame"));
    m_TalkEditFrame->setFrameShape(QFrame::NoFrame);
    m_TalkEditFrame->setFrameShadow(QFrame::Raised);
    m_TalkEditFrame->setLineWidth(0);
    m_gridLayoutTalk = new QGridLayout(m_TalkEditFrame);
    m_gridLayoutTalk->setSpacing(6);
    m_gridLayoutTalk->setContentsMargins(0, 0, 0, 0);
    m_gridLayoutTalk->setObjectName(QString::fromUtf8("m_gridLayoutTalk"));

    m_TalkTitleLabel = new QLabel(m_TalkEditFrame);
    m_TalkTitleLabel->setObjectName(QString::fromUtf8("m_TalkTitleLabel"));
    m_TalkTitleLabel->setText(QString::fromUtf8("Title"));
    m_gridLayoutTalk->addWidget(m_TalkTitleLabel, 0, 0, 1, 1);
    m_TalkTitleEdit = new QLineEdit(m_TalkEditFrame);
    m_TalkTitleEdit->setObjectName(QString::fromUtf8("m_TalkTitleEdit"));
    m_gridLayoutTalk->addWidget(m_TalkTitleEdit, 0, 1, 1, 3);

    m_TalkStartTimeLabel = new QLabel(m_TalkEditFrame);
    m_TalkStartTimeLabel->setObjectName(QString::fromUtf8("m_TalkStartTimeLabel"));
    m_TalkStartTimeLabel->setText(QString::fromUtf8("Start"));
    m_gridLayoutTalk->addWidget(m_TalkStartTimeLabel, 1, 0, 1, 1);
    m_TalkStartTimeEdit = new QTimeEdit(m_AffEditFrame);
    m_TalkStartTimeEdit->setObjectName(QString::fromUtf8("m_TalkStartTimeEdit"));
    m_gridLayoutTalk->addWidget(m_TalkStartTimeEdit, 1, 1, 1, 1);

    m_TalkEndTimeLabel = new QLabel(m_TalkEditFrame);
    m_TalkEndTimeLabel->setObjectName(QString::fromUtf8("m_TalkEndTimeLabel"));
    m_TalkEndTimeLabel->setText(QString::fromUtf8("End"));
    m_TalkEndTimeLabel->setAlignment(Qt::AlignLeading|Qt::AlignRight|Qt::AlignCenter);
    m_gridLayoutTalk->addWidget(m_TalkEndTimeLabel, 1, 2, 1, 1);
    m_TalkEndTimeEdit = new QTimeEdit(m_TalkEditFrame);
    m_TalkEndTimeEdit->setObjectName(QString::fromUtf8("m_TalkEndTimeEdit"));
    m_gridLayoutTalk->addWidget(m_TalkEndTimeEdit, 1, 3, 1, 1);

    m_InvitedTalkAbstractLabel = new QLabel(m_TalkEditFrame);
    m_InvitedTalkAbstractLabel->setObjectName(QString::fromUtf8("m_InvitedTalkAbstractLabel"));
    m_InvitedTalkAbstractLabel->setText(QString::fromUtf8("Abstract"));
    m_InvitedTalkAbstractLabel->setAlignment(Qt::AlignLeading|Qt::AlignLeft|Qt::AlignTop);
    m_gridLayoutTalk->addWidget(m_InvitedTalkAbstractLabel, 2, 0, 1, 1);
    m_InvitedTalkAbstractLabel->hide();
    m_InvitedTalkAbstractEdit = new QPlainTextEdit(m_TalkEditFrame);
    m_InvitedTalkAbstractEdit->setObjectName(QString::fromUtf8("m_InvitedTalkAbstractEdit"));    
    m_gridLayoutTalk->addWidget(m_InvitedTalkAbstractEdit, 2, 1, 1, 3);
    m_InvitedTalkAbstractEdit->hide();

    m_TalkPaperLabel = new QLabel(m_TalkEditFrame);
    m_TalkPaperLabel->setObjectName(QString::fromUtf8("m_TalkPaperLabel"));
    m_TalkPaperLabel->setText(QString::fromUtf8("Paper"));
    m_gridLayoutTalk->addWidget(m_TalkPaperLabel,3, 0, 1, 1);
    m_TalkPaperLabel->hide();
    m_TalkPaperComboBox = new QComboBox(m_TalkEditFrame);
    m_TalkPaperComboBox->setObjectName(QString::fromUtf8("m_TalkPaperComboBox"));
    m_gridLayoutTalk->addWidget(m_TalkPaperComboBox, 3, 1, 1, 3);
    m_TalkPaperComboBox->hide();

    m_TalkSpeakersLabel = new QLabel(m_TalkEditFrame);
    m_TalkSpeakersLabel->setObjectName(QString::fromUtf8("m_TalkSpeakersLabel"));
    m_TalkSpeakersLabel->setText(QString::fromUtf8("Speakers"));
    m_gridLayoutTalk->addWidget(m_TalkSpeakersLabel, 4, 0, 1, 1);

    m_TalkSpeakersList = new QElemList(m_TalkEditFrame, false, false, true);
    m_TalkSpeakersList->setObjectName(QString::fromUtf8("m_TalkSpeakersList"));
    m_gridLayoutTalk->addWidget(m_TalkSpeakersList, 5, 0, 1, 4);

    m_TalkEditSpacer = new QSpacerItem(20, 40, QSizePolicy::Minimum, QSizePolicy::Expanding);
    m_gridLayoutTalk->addItem(m_TalkEditSpacer, 6, 0, 1, 1);

    m_TalkSplitter->addWidget(m_TalkListFrame);
    m_TalkSplitter->addWidget(m_TalkEditFrame);

    QObject::connect(m_TalkTitleEdit, SIGNAL(textEdited(QString)), this, SLOT(talksDataChanged()));
    QObject::connect(m_TalkTitleEdit, SIGNAL(textEdited(QString)), this, SLOT(conferenceDataChanged()));
    QObject::connect(m_TalkStartTimeEdit, SIGNAL(timeChanged (const QTime &)), this, SLOT(talksDataChanged()));
    QObject::connect(m_TalkStartTimeEdit, SIGNAL(timeChanged (const QTime &)), this, SLOT(conferenceDataChanged()));
    QObject::connect(m_TalkEndTimeEdit, SIGNAL(timeChanged (const QTime &)), this, SLOT(talksDataChanged()));
    QObject::connect(m_TalkEndTimeEdit, SIGNAL(timeChanged (const QTime &)), this, SLOT(conferenceDataChanged()));
    QObject::connect(m_InvitedTalkAbstractEdit, SIGNAL(textChanged()), this, SLOT(talksDataChanged()));
    QObject::connect(m_InvitedTalkAbstractEdit, SIGNAL(textChanged()), this, SLOT(conferenceDataChanged()));
    QObject::connect(m_TalkPaperComboBox, SIGNAL(currentIndexChanged(int)), this, SLOT(talksDataChanged()));
    QObject::connect(m_TalkPaperComboBox, SIGNAL(currentIndexChanged(int)), this, SLOT(conferenceDataChanged()));

    QObject::connect(m_TalkList, SIGNAL(newElement()), this, SLOT(talksNewElement()));
    QObject::connect(m_TalkList, SIGNAL(newElement()), this, SLOT(conferenceDataChanged()));
    QObject::connect(m_TalkList, SIGNAL(delElement(index_t)), this, SLOT(talksDelElement(index_t)));
    QObject::connect(m_TalkList, SIGNAL(delElement(index_t)), this, SLOT(conferenceDataChanged()));
    QObject::connect(m_TalkList, SIGNAL(activeElement(index_t)), this, SLOT(talksActiveElement(index_t)));

    QObject::connect(m_TalkSpeakersList, SIGNAL(newElement()), this, SLOT(talksSpeakersNewElement()));
    QObject::connect(m_TalkSpeakersList, SIGNAL(delElement(index_t)), this, SLOT(talksSpeakersDelElement(index_t)));
    QObject::connect(m_TalkSpeakersList, SIGNAL(delElement(index_t)), this, SLOT(conferenceDataChanged()));
    QObject::connect(m_TalkSpeakersList, SIGNAL(upElement(index_t)), this, SLOT(talksSpeakersUpElement(index_t)));
    QObject::connect(m_TalkSpeakersList, SIGNAL(downElement(index_t)), this, SLOT(talksSpeakersDownElement(index_t)));

    QObject::connect(m_TalkImportBtn, SIGNAL(clicked()), this, SLOT(talksImportCSV()));
    QObject::connect(m_TalkTransferBtn, SIGNAL(clicked()), this, SLOT(talksTransferPapers()));

    talksActiveElement(INDEX_EMPTY);
}

void MainWindow::talksImportCSV() {
    if (!SaveCheck("importing talks from CSV file")) return;

    QMessageBox msgBox(QMessageBox::Question,tr("Talk type"), tr("What is the type of the talk that you want to import?"));
    QPushButton *paperButton = msgBox.addButton(tr("Paper Talks"), QMessageBox::ActionRole);
    msgBox.addButton(tr("Invited Talks"), QMessageBox::ActionRole);
    QPushButton *abortButton = msgBox.addButton(QMessageBox::Abort);

    msgBox.exec();
    if (msgBox.clickedButton() == abortButton) return;
    bool bPaper = msgBox.clickedButton() == paperButton;

    std::vector<std::string> properties;
    properties.push_back("Title");
    properties.push_back("Start Minutes");
    properties.push_back("End Minutes");

    if (!bPaper)
        properties.push_back("Abstract");

    QImportDialog d(properties, this);
    if (QDialog::Accepted != d.exec()) return;

    std::string filename = d.GetFilename();
    if (!filename.empty()) {
        std::vector<int> choices = d.GetChoices();
        CSFParser csvData(filename,d.GetOffset(),d.RemoveWhitespaces());
        csvData.SetRecordSizeToMax();

        for (size_t i = 0;i<csvData.RecordCount();i++) {
            std::vector< std::string > csvRecord = csvData.GetRecord(i);

            if (bPaper) {
                PaperTalk pap(choices[0]>= 0 ? csvRecord[choices[0]] : "",
                              choices[1]>= 0 ? atoi(csvRecord[choices[1]].c_str()) : 0,
                              choices[2]>= 0 ? atoi(csvRecord[choices[2]].c_str()) : 0);
                m_pConfDB->talks.AddUniqueEntry(pap);
            } else {
                InvitedTalk inv(choices[0]>= 0 ? csvRecord[choices[0]] : "",
                                choices[1]>= 0 ? atoi(csvRecord[choices[1]].c_str()) : 0,
                                choices[2]>= 0 ? atoi(csvRecord[choices[2]].c_str()) : 0,
                                choices[3]>= 0 ? csvRecord[choices[3]] : "");
                m_pConfDB->talks.AddUniqueEntry(inv);
            }
        }

        conferenceDataChanged();
        UpdateUI();
    }
}

void MainWindow::talksTransferPapers() {
    bool bChanged = false;

    QMessageBox::StandardButton reply;
    reply = QMessageBox::question(this, tr("Speaker"),
                                  tr("Enter the first author as defaut speaker?"),
                                    QMessageBox::Yes | QMessageBox::No | QMessageBox::Cancel);
    if (reply == QMessageBox::Cancel) {
        return;
    }

    bool bFirstAuthorAsSpeaker = (reply == QMessageBox::Yes);

    for (size_t i = 0;i<m_pConfDB->works.Count();++i) {
        const Paper* p = dynamic_cast<const Paper*>(m_pConfDB->works.GetEntryByPos(i));
        if (p && p->GetReferenceCount(m_pConfDB) == 0) {
            PaperTalk newPaperT(p->GetTitle(), 0,0);
            newPaperT.SetPaper(*p);

            if (bFirstAuthorAsSpeaker && p->GetAuthors().size() > 0) {
                const SciPerson* author = m_pConfDB->people.GetEntryByIndex(p->GetAuthors()[0]);
                if (author)
                    newPaperT.AddSpeaker(*author);
            }

            m_pConfDB->talks.AddEntry(newPaperT);
            bChanged = true;
        }
    }
    if (bChanged) {
        conferenceDataChanged();
        UpdatePublicationUI();
        UpdateTalksUI(m_TalkList->GetActiveIndex());
    }
}

void MainWindow::UpdateTalksUI(index_t newIndex) {
    std::vector< ElemListEntry > TalkList;
    for (size_t i = 0;i<m_pConfDB->talks.Count();++i) {
        const Talk* t = m_pConfDB->talks.GetEntryByPos(i);

        const PaperTalk* pt = dynamic_cast<const PaperTalk*>(t);
        if (pt) {
            std::string title = pt->GetTitle();

            if (title.empty() && m_pConfDB->works.GetEntryByIndex(pt->GetPaper()))
                title = m_pConfDB->works.GetEntryByIndex(pt->GetPaper())->GetTitle();

            TalkList.push_back( ElemListEntry(title + std::string(" (Paper)"),
                                                     pt->GetIndex(),
                                                     pt->GetReferenceCount(m_pConfDB))  );
        } else {
            const InvitedTalk* it = dynamic_cast<const InvitedTalk*>(t);
            if (it) {
                TalkList.push_back( ElemListEntry(it->GetTitle() + std::string(" (Invited)"),
                                                         it->GetIndex(),
                                                         it->GetReferenceCount(m_pConfDB))  );
            } else {
                TalkList.push_back( ElemListEntry(t->GetTitle(),
                                                         t->GetIndex(),
                                                         t->GetReferenceCount(m_pConfDB))  );
            }
        }
    }
    m_TalkList->ElemUpdate(TalkList, newIndex);
    UpdateTalksSpeakersList();
}

void MainWindow::UpdateTalksUIEdits() {
    talksActiveElement(m_TalkList->GetActiveIndex());
}

void MainWindow::UpdateTalksSpeakersList() {
    bool b = StartEditBlocker();
    UpdateTalksSpeakersList(m_TalkList->GetActiveIndex());
    EndEditBlocker(b);
}

void MainWindow::UpdateTalksSpeakersList(index_t index) {
    const Talk* t =(m_pConfDB->talks.GetEntryByIndex(index));

    std::vector< ElemListEntry > speakerList;
    if (t) {
        for (size_t i = 0;i<t->GetSpeakers().size();++i) {
            index_t pIndex = t->GetSpeakers()[i];
            const SciPerson* pers = m_pConfDB->people.GetEntryByIndex(pIndex);
            speakerList.push_back( ElemListEntry(pers->GetDisplayName(),
                                                pers->GetIndex()));
        }
    }
    m_TalkSpeakersList->ElemUpdate(speakerList, INDEX_EMPTY);
}

void MainWindow::UpdateTalksPaperDropbox() {
   UpdateTalksPaperDropbox(m_TalkList->GetActiveIndex());
}

void MainWindow::UpdateTalksPaperDropbox(index_t index) {
    const PaperTalk* t = dynamic_cast<PaperTalk*>(m_pConfDB->talks.GetEntryByIndex(index));

    m_TalkPaperComboBox->clear();
    m_TalkPaperComboBox->addItem("Select a paper", 0);
    m_TalkPaperComboBox->setCurrentIndex(0);

    if (!t) return;

    if (m_pConfDB->works.GetEntryByIndex(t->GetPaper())) m_TalkPaperComboBox->clear();

    for (size_t i = 0;i<m_pConfDB->works.Count();++i) {
        const Paper* p = dynamic_cast<Paper*>(m_pConfDB->works.GetEntryByPos(i));
        if (!p) continue;

        // only display papers that are not in use elsewhere
        if (p->GetReferenceCount(m_pConfDB) == 0 || p->GetIndex() == t->GetPaper()) {
            m_TalkPaperComboBox->addItem(QString::fromUtf8(p->GetTitle().c_str()), p->GetIndex());
            if (p->GetIndex() == t->GetPaper())
                m_TalkPaperComboBox->setCurrentIndex(m_TalkPaperComboBox->count()-1);
        }
    }
}

// Slots

void MainWindow::talksDataChanged() {
    if (m_bEditBlockSignals) return;
    index_t TalkIndex = m_TalkList->GetActiveIndex();
    Talk* t = m_pConfDB->talks.GetEntryByIndex(TalkIndex);
    assert(t);

    // common part
    t->SetTitle(std::string(m_TalkTitleEdit->text().toUtf8()));
    t->SetStartTime(m_TalkStartTimeEdit->time().minute()+m_TalkStartTimeEdit->time().hour()*60);
    t->SetEndTime(m_TalkEndTimeEdit->time().minute()+m_TalkEndTimeEdit->time().hour()*60);

    PaperTalk* pt = dynamic_cast<PaperTalk*>(t);
    if (pt) {
        int index= m_TalkPaperComboBox->currentIndex();
        index_t paperIndex = m_TalkPaperComboBox->itemData(index).toUInt();
        const Paper* p = dynamic_cast<const Paper*>(m_pConfDB->works.GetEntryByIndex(paperIndex));

        if (p) {
            if (pt->GetPaper() != paperIndex) {
                const Paper* prevPaper = dynamic_cast<const Paper*>(m_pConfDB->works.GetEntryByIndex(pt->GetPaper()));

                if (t->GetTitle() == "" || t->GetTitle() == "New" ||
                    (prevPaper &&  t->GetTitle() == prevPaper->GetTitle())) {
                    t->SetTitle(p->GetTitle());
                    m_TalkTitleEdit->setText(QString::fromUtf8(t->GetTitle().c_str()));
                }
            }

            pt->SetPaper(*p);
        } else {
            pt->ClearPaper();
        }
        UpdatePublicationUI();
    } else {
        InvitedTalk* it = dynamic_cast<InvitedTalk*>(t);
        if (it) {
            it->SetAbstract(std::string(m_InvitedTalkAbstractEdit->document()->toPlainText().toUtf8()));
        }
    }

    UpdateStartEnd();
    UpdateSessionsUI();
    UpdateTalksUI(TalkIndex);
}

void MainWindow::talksActiveElement(index_t index) {
    bool b = StartEditBlocker();
    const Talk* t =(m_pConfDB->talks.GetEntryByIndex(index));
    if (t) {
        m_TalkTitleEdit->setText(QString::fromUtf8(t->GetTitle().c_str()));
        int startTime = t->GetStartTime();
        m_TalkStartTimeEdit->setTime(QTime(startTime/60, startTime%60));
        int endTime = t->GetEndTime();
        m_TalkEndTimeEdit->setTime(QTime(endTime/60, endTime%60));

        const PaperTalk* pt = dynamic_cast<const PaperTalk*>(t);
        if (pt) {
            m_InvitedTalkAbstractLabel->hide();
            m_InvitedTalkAbstractEdit->hide();

            m_TalkPaperLabel->show();
            m_TalkPaperComboBox->show();
            UpdateTalksPaperDropbox(index);
        } else {
            const InvitedTalk* it = dynamic_cast<const InvitedTalk*>(t);
            if (it) {
                m_TalkPaperLabel->hide();
                m_TalkPaperComboBox->hide();

                m_InvitedTalkAbstractLabel->show();
                m_InvitedTalkAbstractEdit->show();
                m_InvitedTalkAbstractEdit->setPlainText(QString::fromUtf8(it->GetAbstract().c_str()));
            } else {
                m_TalkPaperLabel->hide();
                m_TalkPaperComboBox->hide();
                m_InvitedTalkAbstractLabel->hide();
                m_InvitedTalkAbstractEdit->hide();
            }
        }
        UpdateTalksSpeakersList(index);
        m_TalkTitleEdit->setEnabled(true);
        m_TalkStartTimeEdit->setEnabled(true);
        m_TalkEndTimeEdit->setEnabled(true);
        m_InvitedTalkAbstractEdit->setEnabled(true);
        m_TalkPaperComboBox->setEnabled(true);
        m_TalkSpeakersList->setEnabled(true);

    } else {
        m_TalkTitleEdit->setText("");
        m_TalkStartTimeEdit->setTime(QTime(0, 0));
        m_TalkEndTimeEdit->setTime(QTime(0, 0));
        m_InvitedTalkAbstractEdit->setPlainText("");
        m_TalkPaperComboBox->clear();
        m_TalkSpeakersList->Clear();

        m_TalkPaperLabel->hide();
        m_TalkPaperComboBox->hide();
        m_InvitedTalkAbstractLabel->hide();
        m_InvitedTalkAbstractEdit->hide();

        m_TalkTitleEdit->setEnabled(false);
        m_TalkStartTimeEdit->setEnabled(false);
        m_TalkEndTimeEdit->setEnabled(false);
        m_InvitedTalkAbstractEdit->setEnabled(false);
        m_TalkPaperComboBox->setEnabled(false);
        m_TalkSpeakersList->setEnabled(false);
    }
    EndEditBlocker(b);
}

void MainWindow::talksNewElement() {
    QMessageBox msgBox(QMessageBox::Question,tr("Talk type"), tr("What type of talk do you want to add?"));
    QPushButton *paperButton = msgBox.addButton(tr("Paper Talk"), QMessageBox::ActionRole);
    msgBox.addButton(tr("Invited Talk"), QMessageBox::ActionRole);
    QPushButton *abortButton = msgBox.addButton(QMessageBox::Abort);

    msgBox.exec();
    if (msgBox.clickedButton() == abortButton) return;

    index_t index;
    if (msgBox.clickedButton() == paperButton) {
        PaperTalk newPaperT("New", 0,0);
        index = m_pConfDB->talks.AddEntry(newPaperT);
    } else {
        InvitedTalk newInvT("Invited Talk Title",0,0,"");
        index = m_pConfDB->talks.AddEntry(newInvT);
    }
    UpdateTalksUI(index);
}

void MainWindow::talksDelElement(index_t index) {
    Talk* t = m_pConfDB->talks.GetEntryByIndex(index);
    assert(t);
    assert(t->GetReferenceCount(m_pConfDB) == 0);

    m_pConfDB->talks.RemoveEntryByIndex(index);
    UpdateTalksUI(INDEX_EMPTY);
    UpdatePeopleUI();
}

void MainWindow::talksSpeakersNewElement() {
    QStringList items;
    index_t TalkIndex = m_TalkList->GetActiveIndex();
    Talk* t = m_pConfDB->talks.GetEntryByIndex(TalkIndex);
    assert(t);

    for (size_t i = 0;i<m_pConfDB->people.Count();++i) {
        const SciPerson* pers = m_pConfDB->people.GetEntryByPos(i);
        if (std::find(t->GetSpeakers().begin(), t->GetSpeakers().end(), pers->GetIndex()) == t->GetSpeakers().end())
            items << QString::fromUtf8(pers->GetSortableDisplayName(m_pConfDB).c_str());
    }

    if (items.count() == 0) {
        QMessageBox msgBox;
        msgBox.setIcon(QMessageBox::Warning);
        msgBox.setText("No additional speakers available.");
        msgBox.exec();
        return;
    }

    items.sort();

    bool ok;
    QString item = QInputDialog::getItem(this, tr("Select Additional Speaker"),
                                         tr("Available Speakers:"), items, 0, false, &ok);
    if (ok && !item.isEmpty()) {
        for (size_t i = 0;i<m_pConfDB->people.Count();++i) {
            const SciPerson* pers = m_pConfDB->people.GetEntryByPos(i);
            if (item == QString::fromUtf8(pers->GetSortableDisplayName(m_pConfDB).c_str())) {
                t->AddSpeaker(*pers);
                break;
            }
        }

        UpdatePeopleUI();
        UpdateTalksSpeakersList();
        conferenceDataChanged();
    }
}

void MainWindow::talksSpeakersDelElement(index_t index) {
    index_t TalkIndex = m_TalkList->GetActiveIndex();
    Talk* t = m_pConfDB->talks.GetEntryByIndex(TalkIndex);
    assert(t);

    t->DelSpeaker(index);
    UpdateTalksSpeakersList();
    UpdatePeopleUI();
}

void MainWindow::talksSpeakersUpElement(index_t index) {
    index_t TalkIndex = m_TalkList->GetActiveIndex();
    Talk* t = m_pConfDB->talks.GetEntryByIndex(TalkIndex);
    assert(t);

    t->SpeakerUp(index);
    UpdateTalksSpeakersList();
    conferenceDataChanged();
}

void MainWindow::talksSpeakersDownElement(index_t index) {
    index_t TalkIndex = m_TalkList->GetActiveIndex();
    Talk* t = m_pConfDB->talks.GetEntryByIndex(TalkIndex);
    assert(t);

    t->SpeakerDown(index);
    UpdateTalksSpeakersList();
    conferenceDataChanged();
}
