#include "mainwindow.h"

#include <QtGui/QMessageBox>
#include <QtGui/QInputDialog>
#include <algorithm>

#include "../../CrossPlatform/PaperSession.h"
#include "../../CrossPlatform/PosterSession.h"
#include "../../CrossPlatform/Keynote.h"
#include "../../CrossPlatform/OrganizedSession.h"
#include "../../CrossPlatform/Tutorial.h"
#include "../../CrossPlatform/Panel.h"

void MainWindow::setupSessionsTab(){
    m_tabSessions = new QWidget();
    m_tabSessions->setObjectName(QString::fromUtf8("m_tabSessions"));
    m_tabSessions->setWindowTitle(QString::fromUtf8("Sessions"));
    m_globalTabs->addTab(m_tabSessions, m_tabSessions->windowTitle());

    m_SessionGlobalLayout = new QHBoxLayout(m_tabSessions);
    m_SessionGlobalLayout->setSpacing(6);
    m_SessionGlobalLayout->setContentsMargins(0, 0, 0, 0);
    m_SessionGlobalLayout->setObjectName(QString::fromUtf8("m_SessionGlobalLayout"));
    m_SessionSplitter = new QSplitter();
    m_SessionGlobalLayout->addWidget(m_SessionSplitter);

    m_SessionList = new QElemList(m_tabSessions, true);
    m_SessionList->setObjectName(QString::fromUtf8("m_SessionList"));
    m_SessionSplitter->addWidget(m_SessionList);

    m_SessionEditFrame = new QFrame(m_tabSessions);
    m_SessionEditFrame->setObjectName(QString::fromUtf8("m_SessionEditFrame"));
    m_SessionEditFrame->setFrameShape(QFrame::NoFrame);
    m_SessionEditFrame->setFrameShadow(QFrame::Raised);
    m_SessionEditFrame->setLineWidth(0);
    m_gridLayoutSession = new QGridLayout(m_SessionEditFrame);
    m_gridLayoutSession->setSpacing(6);
    m_gridLayoutSession->setContentsMargins(0, 0, 0, 0);
    m_gridLayoutSession->setObjectName(QString::fromUtf8("m_gridLayoutSession"));

    m_sessionKeynoteRadioButton = new QRadioButton(m_SessionEditFrame);
    m_sessionKeynoteRadioButton->setObjectName(QString::fromUtf8("m_sessionKeynoteRadioButton"));
    m_sessionKeynoteRadioButton->setChecked(true);
    m_sessionKeynoteRadioButton->setText(QString::fromUtf8("Keynote"));
    m_gridLayoutSession->addWidget(m_sessionKeynoteRadioButton, 0, 1, 1, 1);

    m_sessionCapstoneRadioButton = new QRadioButton(m_SessionEditFrame);
    m_sessionCapstoneRadioButton->setObjectName(QString::fromUtf8("m_sessionCapstoneRadioButton"));
    m_sessionCapstoneRadioButton->setChecked(false);
    m_sessionCapstoneRadioButton->setText(QString::fromUtf8("Capstone"));
    m_gridLayoutSession->addWidget(m_sessionCapstoneRadioButton, 0, 2, 1, 1);

    m_SessionTypeLabel = new QLabel(m_SessionEditFrame);
    m_SessionTypeLabel->setObjectName(QString::fromUtf8("m_SessionTypeLabel"));
    m_SessionTypeLabel->setText(QString::fromUtf8("Type"));
    m_gridLayoutSession->addWidget(m_SessionTypeLabel, 0, 0, 1, 1);
    m_SessionTypeEdit = new QLineEdit(m_SessionEditFrame);
    m_SessionTypeEdit->setObjectName(QString::fromUtf8("m_SessionTypeEdit"));
    m_gridLayoutSession->addWidget(m_SessionTypeEdit, 0, 1, 1, 3);

    m_SessionTitleLabel = new QLabel(m_SessionEditFrame);
    m_SessionTitleLabel->setObjectName(QString::fromUtf8("m_SessionTitleLabel"));
    m_SessionTitleLabel->setText(QString::fromUtf8("Title"));
    m_gridLayoutSession->addWidget(m_SessionTitleLabel, 1, 0, 1, 1);
    m_SessionTitleEdit = new QLineEdit(m_SessionEditFrame);
    m_SessionTitleEdit->setObjectName(QString::fromUtf8("m_SessionTitleEdit"));
    m_gridLayoutSession->addWidget(m_SessionTitleEdit, 1, 1, 1, 3);

    m_OrganizedSessionAbstractLabel = new QLabel(m_SessionEditFrame);
    m_OrganizedSessionAbstractLabel->setObjectName(QString::fromUtf8("m_OrganizedSessionAbstractLabel"));
    m_OrganizedSessionAbstractLabel->setText(QString::fromUtf8("Abstract"));
    m_OrganizedSessionAbstractLabel->setAlignment(Qt::AlignLeading|Qt::AlignLeft|Qt::AlignTop);
    m_gridLayoutSession->addWidget(m_OrganizedSessionAbstractLabel, 2, 0, 1, 1);
    m_OrganizedSessionAbstractEdit = new QPlainTextEdit(m_SessionEditFrame);
    m_OrganizedSessionAbstractEdit->setObjectName(QString::fromUtf8("m_OrganizedSessionAbstractEdit"));
    m_gridLayoutSession->addWidget(m_OrganizedSessionAbstractEdit, 2, 1, 1, 3);

    m_SessionURLLabel = new QLabel(m_SessionEditFrame);
    m_SessionURLLabel->setObjectName(QString::fromUtf8("m_SessionURLLabel"));
    m_SessionURLLabel->setText(QString::fromUtf8("URL"));
    m_gridLayoutSession->addWidget(m_SessionURLLabel, 3, 0, 1, 1);
    m_SessionURLEdit = new QLineEdit(m_SessionEditFrame);
    m_SessionURLEdit->setObjectName(QString::fromUtf8("m_SessionURLEdit"));
    m_gridLayoutSession->addWidget(m_SessionURLEdit, 3, 1, 1, 3);

    m_SessionStartTimeLabel = new QLabel(m_SessionEditFrame);
    m_SessionStartTimeLabel->setObjectName(QString::fromUtf8("m_SessionStartTimeLabel"));
    m_SessionStartTimeLabel->setText(QString::fromUtf8("Start"));
    m_gridLayoutSession->addWidget(m_SessionStartTimeLabel, 4, 0, 1, 1);
    m_SessionStartTimeEdit = new QTimeEdit(m_AffEditFrame);
    m_SessionStartTimeEdit->setObjectName(QString::fromUtf8("m_SessionStartTimeEdit"));
    m_gridLayoutSession->addWidget(m_SessionStartTimeEdit, 4, 1, 1, 1);

    m_SessionEndTimeLabel = new QLabel(m_SessionEditFrame);
    m_SessionEndTimeLabel->setObjectName(QString::fromUtf8("m_SessionEndTimeLabel"));
    m_SessionEndTimeLabel->setText(QString::fromUtf8("End"));
    m_SessionEndTimeLabel->setAlignment(Qt::AlignLeading|Qt::AlignRight|Qt::AlignCenter);
    m_gridLayoutSession->addWidget(m_SessionEndTimeLabel, 4, 2, 1, 1);
    m_SessionEndTimeEdit = new QTimeEdit(m_SessionEditFrame);
    m_SessionEndTimeEdit->setObjectName(QString::fromUtf8("m_SessionEndTimeEdit"));
    m_gridLayoutSession->addWidget(m_SessionEndTimeEdit, 4, 3, 1, 1);

    m_SessionChairLabel = new QLabel(m_SessionEditFrame);
    m_SessionChairLabel->setObjectName(QString::fromUtf8("m_SessionChairLabel"));
    m_SessionChairLabel->setText(QString::fromUtf8("Chair"));
    m_gridLayoutSession->addWidget(m_SessionChairLabel,5, 0, 1, 1);
    m_SessionChairComboBox = new QComboBox(m_SessionEditFrame);
    m_SessionChairComboBox->setObjectName(QString::fromUtf8("m_SessionChairComboBox"));
    m_gridLayoutSession->addWidget(m_SessionChairComboBox, 5, 1, 1, 2);

    m_SessionChairMailCheckBox = new QCheckBox(m_SessionEditFrame);
    m_SessionChairMailCheckBox->setObjectName(QString::fromUtf8("m_SessionChairMailCheckBox"));
    m_SessionChairMailCheckBox->setText(QString::fromUtf8("Accepts Mail"));
    m_gridLayoutSession->addWidget(m_SessionChairMailCheckBox, 5, 3, 1, 1);

    m_SessionRoomsLabel = new QLabel(m_SessionEditFrame);
    m_SessionRoomsLabel->setObjectName(QString::fromUtf8("m_SessionRoomsLabel"));
    m_SessionRoomsLabel->setText(QString::fromUtf8("Rooms"));
    m_gridLayoutSession->addWidget(m_SessionRoomsLabel, 6, 0, 1, 1);
    m_SessionRoomsList = new QElemList(m_SessionEditFrame, true, false);
    m_SessionRoomsList->setObjectName(QString::fromUtf8("m_SessionRoomsList"));
    m_gridLayoutSession->addWidget(m_SessionRoomsList, 7, 0, 1, 4);

    m_PaperSessionTalksLabel = new QLabel(m_SessionEditFrame);
    m_PaperSessionTalksLabel->setObjectName(QString::fromUtf8("m_PaperSessionTalksLabel"));
    m_PaperSessionTalksLabel->setText(QString::fromUtf8("Paper Talks"));
    m_gridLayoutSession->addWidget(m_PaperSessionTalksLabel, 8, 0, 1, 1);
    m_PaperSessionTalksList = new QElemList(m_SessionEditFrame, false, false);
    m_PaperSessionTalksList->setObjectName(QString::fromUtf8("m_PaperSessionTalksList"));
    m_gridLayoutSession->addWidget(m_PaperSessionTalksList, 9, 0, 1, 4);

    m_PosterSessionPostersLabel = new QLabel(m_SessionEditFrame);
    m_PosterSessionPostersLabel->setObjectName(QString::fromUtf8("m_PosterSessionPostersLabel"));
    m_PosterSessionPostersLabel->setText(QString::fromUtf8("Posters"));
    m_gridLayoutSession->addWidget(m_PosterSessionPostersLabel, 8, 0, 1, 1);
    m_PosterSessionPostersList = new QElemList(m_SessionEditFrame, false, false);
    m_PosterSessionPostersList->setObjectName(QString::fromUtf8("m_PosterSessionPostersList"));
    m_gridLayoutSession->addWidget(m_PosterSessionPostersList, 9, 0, 1, 4);

    m_InvitedSessionTalksLabel = new QLabel(m_SessionEditFrame);
    m_InvitedSessionTalksLabel->setObjectName(QString::fromUtf8("m_InvitedSessionTalksLabel"));
    m_InvitedSessionTalksLabel->setText(QString::fromUtf8("Talks"));
    m_gridLayoutSession->addWidget(m_InvitedSessionTalksLabel, 8, 0, 1, 1);
    m_InvitedSessionTalksList = new QElemList(m_SessionEditFrame, false, false);
    m_InvitedSessionTalksList->setObjectName(QString::fromUtf8("m_InvitedSessionTalksList"));
    m_gridLayoutSession->addWidget(m_InvitedSessionTalksList, 9, 0, 1, 4);

    m_OrganizedSessionOrganizersLabel = new QLabel(m_SessionEditFrame);
    m_OrganizedSessionOrganizersLabel->setObjectName(QString::fromUtf8("m_OrganizedSessionOrganizersLabel"));
    m_OrganizedSessionOrganizersLabel->setText(QString::fromUtf8("Organizers"));
    m_gridLayoutSession->addWidget(m_OrganizedSessionOrganizersLabel, 8, 0, 1, 1);
    m_OrganizedSessionOrganizersList = new QElemList(m_SessionEditFrame, false, false);
    m_OrganizedSessionOrganizersList->setObjectName(QString::fromUtf8("m_OrganizedSessionOrganizersList"));
    m_gridLayoutSession->addWidget(m_OrganizedSessionOrganizersList, 9, 0, 1, 4);

    m_TutorialSessionTalksLabel = new QLabel(m_SessionEditFrame);
    m_TutorialSessionTalksLabel->setObjectName(QString::fromUtf8("m_TutorialSessionTalksLabel"));
    m_TutorialSessionTalksLabel->setText(QString::fromUtf8("Talks"));
    m_gridLayoutSession->addWidget(m_TutorialSessionTalksLabel, 10, 0, 1, 1);
    m_TutorialSessionTalksList = new QElemList(m_SessionEditFrame, false, false);
    m_TutorialSessionTalksList->setObjectName(QString::fromUtf8("m_TutorialSessionTalksList"));
    m_gridLayoutSession->addWidget(m_TutorialSessionTalksList, 11, 0, 1, 4);

    m_PanelSessionPanelistsLabel = new QLabel(m_SessionEditFrame);
    m_PanelSessionPanelistsLabel->setObjectName(QString::fromUtf8("m_PanelSessionPanelistsLabel"));
    m_PanelSessionPanelistsLabel->setText(QString::fromUtf8("Panelists"));
    m_gridLayoutSession->addWidget(m_PanelSessionPanelistsLabel,10, 0, 1, 1);
    m_PanelSessionPanelistsList = new QElemList(m_SessionEditFrame, false, false);
    m_PanelSessionPanelistsList->setObjectName(QString::fromUtf8("m_PanelSessionPanelistsList"));
    m_gridLayoutSession->addWidget(m_PanelSessionPanelistsList, 11, 0, 1, 4);

    m_SessionEditSpacer = new QSpacerItem(20, 40, QSizePolicy::Minimum, QSizePolicy::Expanding);
    m_gridLayoutSession->addItem(m_SessionEditSpacer, 11, 0, 1, 1);
    m_SessionSplitter->addWidget(m_SessionEditFrame);

    QObject::connect(m_sessionKeynoteRadioButton, SIGNAL(toggled(bool)), this, SLOT(sessionsDataChanged()));
    QObject::connect(m_sessionKeynoteRadioButton, SIGNAL(toggled(bool)), this, SLOT(conferenceDataChanged()));
    QObject::connect(m_SessionTypeEdit, SIGNAL(textEdited(QString)), this, SLOT(sessionsDataChanged()));
    QObject::connect(m_SessionTypeEdit, SIGNAL(textEdited(QString)), this, SLOT(conferenceDataChanged()));
    QObject::connect(m_SessionTitleEdit, SIGNAL(textEdited(QString)), this, SLOT(sessionsDataChanged()));
    QObject::connect(m_SessionTitleEdit, SIGNAL(textEdited(QString)), this, SLOT(conferenceDataChanged()));
    QObject::connect(m_SessionURLEdit, SIGNAL(textEdited(QString)), this, SLOT(sessionsDataChanged()));
    QObject::connect(m_SessionURLEdit, SIGNAL(textEdited(QString)), this, SLOT(conferenceDataChanged()));
    QObject::connect(m_SessionStartTimeEdit, SIGNAL(timeChanged (const QTime &)), this, SLOT(sessionsDataChanged()));
    QObject::connect(m_SessionStartTimeEdit, SIGNAL(timeChanged (const QTime &)), this, SLOT(conferenceDataChanged()));
    QObject::connect(m_SessionEndTimeEdit, SIGNAL(timeChanged (const QTime &)), this, SLOT(sessionsDataChanged()));
    QObject::connect(m_SessionEndTimeEdit, SIGNAL(timeChanged (const QTime &)), this, SLOT(conferenceDataChanged()));
    QObject::connect(m_SessionChairComboBox, SIGNAL(currentIndexChanged(int)), this, SLOT(sessionsDataChanged()));
    QObject::connect(m_SessionChairComboBox, SIGNAL(currentIndexChanged(int)), this, SLOT(conferenceDataChanged()));
    QObject::connect(m_SessionChairMailCheckBox, SIGNAL(stateChanged(int)), this, SLOT(sessionsDataChanged()));
    QObject::connect(m_SessionChairMailCheckBox, SIGNAL(stateChanged(int)), this, SLOT(conferenceDataChanged()));
    QObject::connect(m_OrganizedSessionAbstractEdit, SIGNAL(textChanged()), this, SLOT(sessionsDataChanged()));
    QObject::connect(m_OrganizedSessionAbstractEdit, SIGNAL(textChanged()), this, SLOT(conferenceDataChanged()));

    QObject::connect(m_SessionList, SIGNAL(newElement()), this, SLOT(sessionsNewElement()));
    QObject::connect(m_SessionList, SIGNAL(delElement(index_t)), this, SLOT(sessionsDelElement(index_t)));
    QObject::connect(m_SessionList, SIGNAL(delElement(index_t)), this, SLOT(conferenceDataChanged()));
    QObject::connect(m_SessionList, SIGNAL(activeElement(index_t)), this, SLOT(sessionsActiveElement(index_t)));

    QObject::connect(m_SessionRoomsList, SIGNAL(newElement()), this, SLOT(sessionRoomsNewElement()));
    QObject::connect(m_SessionRoomsList, SIGNAL(delElement(index_t)), this, SLOT(sessionRoomsDelElement(index_t)));
    QObject::connect(m_SessionRoomsList, SIGNAL(delElement(index_t)), this, SLOT(conferenceDataChanged()));

    QObject::connect(m_PaperSessionTalksList, SIGNAL(newElement()), this, SLOT(sessionTalkListNewElement()));
    QObject::connect(m_PaperSessionTalksList, SIGNAL(delElement(index_t)), this, SLOT(sessionTalkListDelElement(index_t)));
    QObject::connect(m_PaperSessionTalksList, SIGNAL(delElement(index_t)), this, SLOT(conferenceDataChanged()));

    QObject::connect(m_PosterSessionPostersList, SIGNAL(newElement()), this, SLOT(posterSessionPostersNewElement()));
    QObject::connect(m_PosterSessionPostersList, SIGNAL(delElement(index_t)), this, SLOT(posterSessionPostersDelElement(index_t)));
    QObject::connect(m_PosterSessionPostersList, SIGNAL(delElement(index_t)), this, SLOT(conferenceDataChanged()));

    QObject::connect(m_InvitedSessionTalksList, SIGNAL(newElement()), this, SLOT(invitedSessionTalksNewElement()));
    QObject::connect(m_InvitedSessionTalksList, SIGNAL(delElement(index_t)), this, SLOT(invitedSessionTalksDelElement(index_t)));
    QObject::connect(m_InvitedSessionTalksList, SIGNAL(delElement(index_t)), this, SLOT(conferenceDataChanged()));

    QObject::connect(m_OrganizedSessionOrganizersList, SIGNAL(newElement()), this, SLOT(organizedSessionOrganizersNewElement()));
    QObject::connect(m_OrganizedSessionOrganizersList, SIGNAL(delElement(index_t)), this, SLOT(organizedSessionOrganizersDelElement(index_t)));
    QObject::connect(m_OrganizedSessionOrganizersList, SIGNAL(delElement(index_t)), this, SLOT(conferenceDataChanged()));

    QObject::connect(m_TutorialSessionTalksList, SIGNAL(newElement()), this, SLOT(tutorialSessionTalksNewElement()));
    QObject::connect(m_TutorialSessionTalksList, SIGNAL(delElement(index_t)), this, SLOT(tutorialSessionTalksDelElement(index_t)));
    QObject::connect(m_TutorialSessionTalksList, SIGNAL(delElement(index_t)), this, SLOT(conferenceDataChanged()));

    QObject::connect(m_PanelSessionPanelistsList, SIGNAL(newElement()), this, SLOT(panelSessionPanelistsNewElement()));
    QObject::connect(m_PanelSessionPanelistsList, SIGNAL(delElement(index_t)), this, SLOT(panelSessionPanelistsDelElement(index_t)));
    QObject::connect(m_PanelSessionPanelistsList, SIGNAL(delElement(index_t)), this, SLOT(conferenceDataChanged()));

    sessionsActiveElement(INDEX_EMPTY);
}

void MainWindow::UpdateStartEnd() {
    UpdateStartEnd(m_SessionList->GetActiveIndex());
}

void MainWindow::UpdateStartEnd(index_t index) {

    Session* s = m_pConfDB->sessions.GetEntryByIndex(index);

    if (!s) return;

    bool bChangesMade = false;
    std::string sessionName = s->GetClassNameViaObject();
    if (sessionName ==PaperSession::GetClassName()) {
        PaperSession* p = dynamic_cast<PaperSession*>(s);

        for (size_t i = 0;i<p->GetPaperTalks().size();++i) {
            const Talk* talk = m_pConfDB->talks.GetEntryByIndex(p->GetPaperTalks()[i]);
            if (talk->GetStartTime() < s->GetStartTime()) {
                s->SetStartTime(talk->GetStartTime());
                bChangesMade = true;
            }

            if (talk->GetEndTime() > s->GetEndTime()) {
                s->SetEndTime(talk->GetEndTime());
                bChangesMade = true;
            }
        }
    }
    if (sessionName ==Keynote::GetClassName()) {
        Keynote* k = dynamic_cast<Keynote*>(s);

        for (size_t i = 0;i<k->GetInvitedTalks().size();++i) {
            const Talk* talk = m_pConfDB->talks.GetEntryByIndex(k->GetInvitedTalks()[i]);
            if (talk->GetStartTime() < s->GetStartTime()) {
                s->SetStartTime(talk->GetStartTime());
                bChangesMade = true;
            }

            if (talk->GetEndTime() > s->GetEndTime()) {
                s->SetEndTime(talk->GetEndTime());
                bChangesMade = true;
            }
        }
    }
    if (sessionName ==Tutorial::GetClassName()) {
        Tutorial* t = dynamic_cast<Tutorial*>(s);

        for (size_t i = 0;i<t->GetTutorialTalks().size();++i) {
            index_t index = t->GetTutorialTalks()[i];
            const Talk* talk = m_pConfDB->talks.GetEntryByIndex(index);
            if (talk->GetStartTime() < s->GetStartTime()) {
                s->SetStartTime(talk->GetStartTime());
                bChangesMade = true;
            }

            if (talk->GetEndTime() > s->GetEndTime()) {
                s->SetEndTime(talk->GetEndTime());
                bChangesMade = true;
            }
        }
    }

    if (bChangesMade) {
        sessionsActiveElement(index);
    }

}


void MainWindow::UpdateSessionsUI(index_t newIndex) {
    std::vector< ElemListEntry > sessionList;
    for (size_t i = 0;i<m_pConfDB->sessions.Count();++i) {
        const Session* s  = m_pConfDB->sessions.GetEntryByPos(i);
        sessionList.push_back( ElemListEntry(s->GetDescription() + std::string(": ") + s->GetTitle(),
                                           s->GetIndex(),
                                           s->GetReferenceCount(m_pConfDB))  );
    }
    m_SessionList->ElemUpdate(sessionList, newIndex);

    UpdateSessionLists();
}

void MainWindow::UpdateSessionLists() {
    UpdateSessionLists(m_SessionList->GetActiveIndex());
}

void MainWindow::UpdateSessionLists(index_t index) {
    const Session* s =  m_pConfDB->sessions.GetEntryByIndex(index);

    std::string sessionName;
    std::vector< ElemListEntry > elemList;
    if (s) {
        for (size_t i = 0;i<s->GetRooms().size();++i) {
            index_t rIndex = s->GetRooms()[i];
            const Room* room = m_pConfDB->rooms.GetEntryByIndex(rIndex);

            elemList.push_back( ElemListEntry(room->GetDisplayName(m_pConfDB),
                                              room->GetIndex()));
        }
        sessionName = s->GetClassNameViaObject();
    }
    m_SessionRoomsList->ElemUpdate(elemList, INDEX_EMPTY);

    if (sessionName ==PaperSession::GetClassName()) {
        const PaperSession* p = dynamic_cast<const PaperSession*>(s);

        std::vector< ElemListEntry > elemList;
        if (p) {
            for (size_t i = 0;i<p->GetPaperTalks().size();++i) {
                index_t tIndex = p->GetPaperTalks()[i];
                const PaperTalk* talk = dynamic_cast<const PaperTalk*>(m_pConfDB->talks.GetEntryByIndex(tIndex));

                std::string title = talk->GetTitle();
                if (title.empty() && m_pConfDB->works.GetEntryByIndex(talk->GetPaper()))
                    title = m_pConfDB->works.GetEntryByIndex(talk->GetPaper())->GetTitle();


                int startTime = talk->GetStartTime();
                int endTime = talk -> GetEndTime();
                title += std::string(" - ") + std::string(QTime(startTime/60, startTime%60).toString("H:mm").toUtf8()) +
                         std::string(" - ") + std::string(QTime(endTime/60, endTime%60).toString("H:mm").toUtf8());

                elemList.push_back( ElemListEntry(title,
                                                  talk->GetIndex()));
            }
        }
        m_PaperSessionTalksList->ElemUpdate(elemList, INDEX_EMPTY);
    }

    if (sessionName ==PosterSession::GetClassName()) {
        const PosterSession* p = dynamic_cast<const PosterSession*>(s);

        std::vector< ElemListEntry > elemList;
        if (p) {
            for (size_t i = 0;i<p->GetPosters().size();++i) {
                index_t pIndex = p->GetPosters()[i];
                const Poster* poster = dynamic_cast<const Poster*>(m_pConfDB->works.GetEntryByIndex(pIndex));

                elemList.push_back( ElemListEntry(poster->GetTitle(),
                                                  poster->GetIndex()));
            }
        }
        m_PosterSessionPostersList->ElemUpdate(elemList, INDEX_EMPTY);
    }

    if (sessionName ==Keynote::GetClassName()) {
        const Keynote* k = dynamic_cast<const Keynote*>(s);

        std::vector< ElemListEntry > elemList;
        if (k) {
            for (size_t i = 0;i<k->GetInvitedTalks().size();++i) {
                index_t tIndex = k->GetInvitedTalks()[i];
                const InvitedTalk* talk = dynamic_cast<const InvitedTalk*>(m_pConfDB->talks.GetEntryByIndex(tIndex));

                std::string title = talk->GetTitle();
                int startTime = talk->GetStartTime();
                int endTime = talk -> GetEndTime();
                title += std::string(" - ") + std::string(QTime(startTime/60, startTime%60).toString("H:mm").toUtf8()) +
                         std::string(" - ") + std::string(QTime(endTime/60, endTime%60).toString("H:mm").toUtf8());

                elemList.push_back( ElemListEntry(title,
                                                  talk->GetIndex()));
            }
        }
        m_InvitedSessionTalksList->ElemUpdate(elemList, INDEX_EMPTY);
    }

    if (sessionName == OrganizedSession::GetClassName() ||
        sessionName == Tutorial::GetClassName() ||
        sessionName == Panel::GetClassName()) {

        const OrganizedSession* os = dynamic_cast<const OrganizedSession*>(s);

        std::vector< ElemListEntry > elemList;
        if (os) {
            for (size_t i = 0;i<os->GetOrganizers().size();++i) {
                index_t oIndex = os->GetOrganizers()[i];
                const SciPerson* organizer = m_pConfDB->people.GetEntryByIndex(oIndex);

                elemList.push_back( ElemListEntry(organizer->GetDisplayName(),
                                                  organizer->GetIndex()));
            }
        }
        m_OrganizedSessionOrganizersList->ElemUpdate(elemList, INDEX_EMPTY);
    }

    if (sessionName ==Tutorial::GetClassName()) {
        const Tutorial* t = dynamic_cast<const Tutorial*>(s);
        std::vector< ElemListEntry > elemList;
        if (t) {
            for (size_t i = 0;i<t->GetTutorialTalks().size();++i) {
                index_t tIndex = t->GetTutorialTalks()[i];
                const InvitedTalk* talk = dynamic_cast<const InvitedTalk*>(m_pConfDB->talks.GetEntryByIndex(tIndex));

                std::string title = talk->GetTitle();
                int startTime = talk->GetStartTime();
                int endTime = talk -> GetEndTime();
                title += std::string(" - ") + std::string(QTime(startTime/60, startTime%60).toString("H:mm").toUtf8()) +
                         std::string(" - ") + std::string(QTime(endTime/60, endTime%60).toString("H:mm").toUtf8());

                elemList.push_back( ElemListEntry(title,
                                                  talk->GetIndex()));
            }
        }
        m_TutorialSessionTalksList->ElemUpdate(elemList, INDEX_EMPTY);
    }

    if (sessionName ==Panel::GetClassName()) {
        const Panel* p = dynamic_cast<const Panel*>(s);

        std::vector< ElemListEntry > elemList;
        if (p) {
            for (size_t i = 0;i<p->GetPanelists().size();++i) {
                index_t pIndex = p->GetPanelists()[i];
                const SciPerson* panelist = m_pConfDB->people.GetEntryByIndex(pIndex);

                elemList.push_back( ElemListEntry(panelist->GetDisplayName(),
                                                  panelist->GetIndex()));
            }
        }
        m_PanelSessionPanelistsList->ElemUpdate(elemList, INDEX_EMPTY);
    }
}

void MainWindow::SessionsFillChairCB() {
    bool b = StartEditBlocker();
    SessionsFillChairCB(m_SessionList->GetActiveIndex());
    EndEditBlocker(b);
}

void MainWindow::SessionsFillChairCB(index_t index) {
    const Session* s =  m_pConfDB->sessions.GetEntryByIndex(index);

    m_SessionChairComboBox->clear();

    if (!s) return;

    for (size_t i = 0;i< m_pConfDB->people.Count();++i) {
        const SciPerson* p =  m_pConfDB->people.GetEntryByPos(i);

        m_SessionChairComboBox->addItem(QString::fromUtf8(p->GetSortableDisplayName(m_pConfDB).c_str()), p->GetIndex());
        if (p->GetIndex() == s->GetSessionChair())
            m_SessionChairComboBox->setCurrentIndex(m_SessionChairComboBox->count()-1);
    }

    m_SessionChairComboBox->model()->sort(0);
    m_SessionChairComboBox->addItem("None", 0);
    if (0 == s->GetSessionChair())
        m_SessionChairComboBox->setCurrentIndex(m_SessionChairComboBox->count()-1);
}

// Slots

void MainWindow::sessionsDataChanged() {
    if (m_bEditBlockSignals) return;
    index_t sIndex = m_SessionList->GetActiveIndex();

    Session* s = m_pConfDB->sessions.GetEntryByIndex(sIndex);
    s->SetTitle(std::string(m_SessionTitleEdit->text().toUtf8()));
    int t = m_SessionStartTimeEdit->time().hour()*60+m_SessionStartTimeEdit->time().minute();
    s->SetStartTime(t);
    t = m_SessionEndTimeEdit->time().hour()*60+m_SessionEndTimeEdit->time().minute();
    s->SetEndTime(t);
    s->SetURL(std::string(m_SessionURLEdit->text().toUtf8()));

    int cbIndex= m_SessionChairComboBox->currentIndex();
    index_t personIndex = m_SessionChairComboBox->itemData(cbIndex).toUInt();
    const SciPerson* p = m_pConfDB->people.GetEntryByIndex(personIndex);
    if (p)
        s->SetSessionChair(*p);
    else
        s->ClearSessionChair();
    UpdatePeopleUI();

    s->SetChairAcceptsMail(m_SessionChairMailCheckBox->isChecked());

    std::string sessionName = s->GetClassNameViaObject();
    if (sessionName ==Keynote::GetClassName()) {
        Keynote* k = dynamic_cast<Keynote*>(s);
        k->SetIsCapstone(m_sessionCapstoneRadioButton->isChecked());
    }

    if (sessionName == OrganizedSession::GetClassName() ||
        sessionName == Tutorial::GetClassName() ||
        sessionName == Panel::GetClassName()) {
        OrganizedSession* os = dynamic_cast<OrganizedSession*>(s);
        os->SetAbstract(std::string(m_OrganizedSessionAbstractEdit->document()->toPlainText().toUtf8()));
    }

    if (sessionName == OrganizedSession::GetClassName() ||
        sessionName == Session::GetClassName()) {
        s->SetDescription(std::string(m_SessionTypeEdit->text().toUtf8()));
    }

    UpdateTracksUI();
    UpdateSessionsUI(sIndex);
}

void MainWindow::sessionsActiveElement(index_t index) {
    bool b = StartEditBlocker();
    const Session* s = m_pConfDB->sessions.GetEntryByIndex(index);

    m_SessionTypeLabel->hide();
    m_SessionTypeEdit->hide();
    m_sessionKeynoteRadioButton->hide();
    m_sessionCapstoneRadioButton->hide();
    m_OrganizedSessionAbstractLabel->hide();
    m_OrganizedSessionAbstractEdit->hide();
    m_PaperSessionTalksLabel->hide();
    m_PaperSessionTalksList->hide();
    m_PosterSessionPostersLabel->hide();
    m_PosterSessionPostersList->hide();
    m_InvitedSessionTalksLabel->hide();
    m_InvitedSessionTalksList->hide();
    m_OrganizedSessionOrganizersLabel->hide();
    m_OrganizedSessionOrganizersList->hide();
    m_TutorialSessionTalksLabel->hide();
    m_TutorialSessionTalksList->hide();
    m_PanelSessionPanelistsLabel->hide();
    m_PanelSessionPanelistsList->hide();

    if (s) {
        m_SessionTitleEdit->setText(QString::fromUtf8(s->GetTitle().c_str()));
        int startTime = s->GetStartTime();
        m_SessionStartTimeEdit->setTime(QTime(startTime/60, startTime%60));
        int endTime = s->GetEndTime();
        m_SessionEndTimeEdit->setTime(QTime(endTime/60, endTime%60));
        m_SessionURLEdit->setText(QString::fromUtf8(s->GetURL().c_str()));
        SessionsFillChairCB(index);
        m_SessionChairMailCheckBox->setChecked(s->GetChairAcceptsMail());

        // we could use RTTI here but that would not work with the
        // session check at the bottom (were we only want the
        // plain sessions and not the derived classes) so for
        // uniforimty we check the strings for all cases
        std::string sessionName = s->GetClassNameViaObject();
        if (sessionName ==PaperSession::GetClassName()) {
            m_PaperSessionTalksLabel->show();
            m_PaperSessionTalksList->show();
        }

        if (sessionName ==PosterSession::GetClassName()) {
            m_PosterSessionPostersLabel->show();
            m_PosterSessionPostersList->show();
        }

        if (sessionName ==Keynote::GetClassName()) {
            const Keynote* k = dynamic_cast<const Keynote*>(s);

            m_sessionCapstoneRadioButton->setChecked(k->GetIsCapstone());
            m_sessionKeynoteRadioButton->setChecked(!k->GetIsCapstone());
            m_sessionKeynoteRadioButton->show();
            m_sessionCapstoneRadioButton->show();
            m_InvitedSessionTalksLabel->show();
            m_InvitedSessionTalksList->show();
        }

        if (sessionName == OrganizedSession::GetClassName() ||
            sessionName == Tutorial::GetClassName() ||
            sessionName == Panel::GetClassName()) {
            m_OrganizedSessionAbstractLabel->show();
            m_OrganizedSessionAbstractEdit->show();

            m_OrganizedSessionOrganizersLabel->show();
            m_OrganizedSessionOrganizersList->show();

            const OrganizedSession* os = dynamic_cast<const OrganizedSession*>(s);
            m_OrganizedSessionAbstractEdit->setPlainText(QString::fromUtf8(os->GetAbstract().c_str()));
        }

        if (sessionName ==Tutorial::GetClassName()) {
            m_TutorialSessionTalksLabel->show();
            m_TutorialSessionTalksList->show();
        }

        if (sessionName ==Panel::GetClassName()) {
            m_PanelSessionPanelistsLabel->show();
            m_PanelSessionPanelistsList->show();
        }

        if (sessionName == OrganizedSession::GetClassName() ||
            sessionName ==Session::GetClassName()) {
          m_SessionTypeEdit->show();
          m_SessionTypeLabel->show();
          m_SessionTypeEdit->setText(QString::fromUtf8(s->GetDescription().c_str()));
        }


        UpdateSessionLists(index);

        m_SessionTitleEdit->setEnabled(true);
        m_SessionStartTimeEdit->setEnabled(true);
        m_SessionEndTimeEdit->setEnabled(true);
        m_SessionURLEdit->setEnabled(true);
        m_SessionChairComboBox->setEnabled(true);
        m_SessionChairMailCheckBox->setEnabled(true);
        m_OrganizedSessionAbstractEdit->setEnabled(true);
        m_SessionTypeEdit->setEnabled(true);
        m_sessionKeynoteRadioButton->setEnabled(true);
        m_sessionCapstoneRadioButton->setEnabled(true);
        m_SessionList->setEnabled(true);
        m_SessionRoomsList->setEnabled(true);
        m_PaperSessionTalksList->setEnabled(true);
        m_PosterSessionPostersList->setEnabled(true);
        m_InvitedSessionTalksList->setEnabled(true);
        m_OrganizedSessionOrganizersList->setEnabled(true);
        m_TutorialSessionTalksList->setEnabled(true);
        m_PanelSessionPanelistsList->setEnabled(true);
    } else {
        m_SessionTypeEdit->setText("");
        m_SessionTitleEdit->setText("");
        m_SessionStartTimeEdit->setTime(QTime(0,0));
        m_SessionEndTimeEdit->setTime(QTime(0,0));
        m_SessionURLEdit->setText("");
        m_SessionChairComboBox->clear();
        m_SessionChairMailCheckBox->setChecked(false);
        m_sessionCapstoneRadioButton->setChecked(false);
        m_OrganizedSessionAbstractEdit->setPlainText("");

        m_SessionList->Clear();
        m_SessionRoomsList->Clear();
        m_PaperSessionTalksList->Clear();
        m_PosterSessionPostersList->Clear();
        m_InvitedSessionTalksList->Clear();
        m_OrganizedSessionOrganizersList->Clear();
        m_TutorialSessionTalksList->Clear();
        m_PanelSessionPanelistsList->Clear();

        m_SessionTitleEdit->setEnabled(false);
        m_SessionStartTimeEdit->setEnabled(false);
        m_SessionEndTimeEdit->setEnabled(false);
        m_SessionURLEdit->setEnabled(false);
        m_SessionChairComboBox->setEnabled(false);
        m_SessionChairMailCheckBox->setEnabled(false);
        m_OrganizedSessionAbstractEdit->setEnabled(false);
        m_SessionTypeEdit->setEnabled(false);
        m_sessionKeynoteRadioButton->setEnabled(false);
        m_sessionCapstoneRadioButton->setEnabled(false);
        m_SessionRoomsList->setEnabled(false);
        m_PaperSessionTalksList->setEnabled(false);
        m_PosterSessionPostersList->setEnabled(false);
        m_InvitedSessionTalksList->setEnabled(false);
        m_OrganizedSessionOrganizersList->setEnabled(false);
        m_TutorialSessionTalksList->setEnabled(false);
        m_PanelSessionPanelistsList->setEnabled(false);
    }
    EndEditBlocker(b);
}


void MainWindow::sessionsNewElement() {
    QMessageBox msgBox(QMessageBox::Question,tr("Session type"), tr("What type of session do you want to add?\n\n"
                                                                    "Paper Session: Papers talks are presented in this session\n"
                                                                    "Poster Session: Posters are presented in this session\n"
                                                                    "Keynote/Capstone: A number of invited talks are presented in this session\n"
                                                                    "Tutorial: Invited Tutorial Talks are presented in this session, which is organized by organizers\n"
                                                                    "Panel: A number of panelists talk about a given topic in this session, which is organized by organizers\n"
                                                                    "Organized (Session): A session, which is organized by organizers (e.g. BoF)\n"
                                                                    "Other (Session): A general session (e.g. Break, Social Event)"
                                                                    ));
    QPushButton *paperButton = msgBox.addButton(tr("Paper Session"), QMessageBox::ActionRole);
    QPushButton *posterButton = msgBox.addButton(tr("Poster Session"), QMessageBox::ActionRole);
    QPushButton *keynoteButton = msgBox.addButton(tr("Keynote/Capstone"), QMessageBox::ActionRole);
    QPushButton *tutorialButton = msgBox.addButton(tr("Tutorial"), QMessageBox::ActionRole);
    QPushButton *panelButton = msgBox.addButton(tr("Panel"), QMessageBox::ActionRole);
    QPushButton *orgButton = msgBox.addButton(tr("Organized"), QMessageBox::ActionRole);
    QPushButton *otherButton = msgBox.addButton(tr("Other"), QMessageBox::ActionRole);
    QPushButton *abortButton = msgBox.addButton(QMessageBox::Abort);

    msgBox.exec();
    if (msgBox.clickedButton() == abortButton) return;

    index_t index = 0;
    if (msgBox.clickedButton() == paperButton) {
        PaperSession newPaperSession("New", 0,0);
        index = m_pConfDB->sessions.AddEntry(newPaperSession);
    }

    if (msgBox.clickedButton() == posterButton) {
        PosterSession newPosterSession("New", 0,0);
        index = m_pConfDB->sessions.AddEntry(newPosterSession);
    }

    if (msgBox.clickedButton() == keynoteButton) {
        Keynote newKeynote("New", 0,0, false);
        index = m_pConfDB->sessions.AddEntry(newKeynote);
    }

    if (msgBox.clickedButton() == orgButton) {
        OrganizedSession newOrgSession("New", 0,0, "Organized","");
        index = m_pConfDB->sessions.AddEntry(newOrgSession);
    }

    if (msgBox.clickedButton() == tutorialButton) {
        Tutorial newTut("New", 0,0,"");
        index = m_pConfDB->sessions.AddEntry(newTut);
    }

    if (msgBox.clickedButton() == panelButton) {
        Panel newPanel("New", 0,0,"");
        index = m_pConfDB->sessions.AddEntry(newPanel);
    }

    if (msgBox.clickedButton() == otherButton) {
        Session newSession("New", 0,0, "Unknown");
        index = m_pConfDB->sessions.AddEntry(newSession);
    }

    m_pConfDB->sessions.GetEntryByIndex(index)->SetChairAcceptsMail(m_pConfDB->GetSupportsSessionMails());

    UpdateSessionsUI(index);
    conferenceDataChanged();
}

void MainWindow::sessionsDelElement(index_t index) {
    Session* s = m_pConfDB->sessions.GetEntryByIndex(index);
    assert(s);
    assert(s->GetReferenceCount(m_pConfDB) == 0);

    m_pConfDB->sessions.RemoveEntryByIndex(index);
    UpdateSessionsUI(INDEX_EMPTY);

    UpdatePeopleUI();
    UpdateTalksUI();
    UpdatePublicationUI();
    UpdateRoomsUI();
}

void MainWindow::sessionRoomsNewElement() {
    QStringList items;
    index_t sIndex = m_SessionList->GetActiveIndex();
    Session* s = m_pConfDB->sessions.GetEntryByIndex(sIndex);
    assert(s);

    for (size_t i = 0;i<m_pConfDB->rooms.Count();++i) {
        const Room* room = m_pConfDB->rooms.GetEntryByPos(i);

        if (std::find(s->GetRooms().begin(), s->GetRooms().end(), room->GetIndex()) == s->GetRooms().end())
            items << QString::fromUtf8(room->GetDisplayName(m_pConfDB).c_str());
    }

    if (items.count() == 0) {
        QMessageBox msgBox;
        msgBox.setIcon(QMessageBox::Warning);
        msgBox.setText("No additional rooms available.");
        msgBox.exec();
        return;
    }

    items.sort();

    bool ok;
    QString item = QInputDialog::getItem(this, tr("Select Room"),
                                         tr("Available rooms:"), items, 0, false, &ok);
    if (ok && !item.isEmpty()) {
        for (size_t i = 0;i<m_pConfDB->rooms.Count();++i) {
            const Room* room = m_pConfDB->rooms.GetEntryByPos(i);

            if (item == QString::fromUtf8(room->GetDisplayName(m_pConfDB).c_str())) {
                s->AddRoom(*room);
                break;
            }
        }

        UpdateSessionsUI(sIndex);
        UpdateRoomsUI();
        conferenceDataChanged();
    }
}

void MainWindow::sessionRoomsDelElement(index_t index) {
    index_t sIndex = m_SessionList->GetActiveIndex();
    Session* s = m_pConfDB->sessions.GetEntryByIndex(sIndex);

    s->DelRoom(index);

    UpdateSessionsUI(sIndex);
    UpdateRoomsUI();
}

void MainWindow::sessionTalkListNewElement() {
    QStringList items;
    index_t sIndex = m_SessionList->GetActiveIndex();
    PaperSession* p = dynamic_cast<PaperSession*>(m_pConfDB->sessions.GetEntryByIndex(sIndex));
    assert(p);

    for (size_t i = 0;i<m_pConfDB->talks.Count();++i) {
        const PaperTalk* talk = dynamic_cast<const PaperTalk*>(m_pConfDB->talks.GetEntryByPos(i));
        if (!talk) continue;

        std::string title = talk->GetTitle();
        if (title.empty() && m_pConfDB->works.GetEntryByIndex(talk->GetPaper()))
            title = m_pConfDB->works.GetEntryByIndex(talk->GetPaper())->GetTitle();

        if (talk->GetReferenceCount(m_pConfDB) == 0)
            items << QString::fromUtf8((title + std::string(" - ") + talk->GetDisplaySpeakers(m_pConfDB)).c_str());
    }

    if (items.count() == 0) {
        QMessageBox msgBox;
        msgBox.setIcon(QMessageBox::Warning);
        msgBox.setText("No additional paper talks available.");
        msgBox.exec();
        return;
    }

    items.sort();

    bool ok;
    QString item = QInputDialog::getItem(this, tr("Select Talk"),
                                         tr("Available paper talks:"), items, 0, false, &ok);
    if (ok && !item.isEmpty()) {
        for (size_t i = 0;i<m_pConfDB->talks.Count();++i) {
            const PaperTalk* talk = dynamic_cast<const PaperTalk*>(m_pConfDB->talks.GetEntryByPos(i));
            if (!talk) continue;

            std::string title = talk->GetTitle();
            if (title.empty() && m_pConfDB->works.GetEntryByIndex(talk->GetPaper()))
                title = m_pConfDB->works.GetEntryByIndex(talk->GetPaper())->GetTitle();

            if (item == QString::fromUtf8((title + std::string(" - ") + talk->GetDisplaySpeakers(m_pConfDB)).c_str())) {
                p->AddPaperTalk(*talk);
                break;
            }
        }

        UpdateStartEnd(sIndex);
        UpdateSessionsUI(sIndex);
        UpdateTalksUI();
        conferenceDataChanged();
    }
}

void MainWindow::sessionTalkListDelElement(index_t index) {
    index_t sIndex = m_SessionList->GetActiveIndex();
    PaperSession* p = dynamic_cast<PaperSession*>(m_pConfDB->sessions.GetEntryByIndex(sIndex));

    p->DelPaperTalk(index);

    UpdateSessionsUI(sIndex);
    UpdateTalksUI();
}

void MainWindow::posterSessionPostersNewElement(){
    QStringList items;
    index_t sIndex = m_SessionList->GetActiveIndex();
    PosterSession* p = dynamic_cast<PosterSession*>(m_pConfDB->sessions.GetEntryByIndex(sIndex));
    assert(p);

    for (size_t i = 0;i<m_pConfDB->works.Count();++i) {
        const Poster* poster = dynamic_cast<const Poster*>(m_pConfDB->works.GetEntryByPos(i));
        if (!poster) continue;

        if (std::find(p->GetPosters().begin(), p->GetPosters().end(), poster->GetIndex()) == p->GetPosters().end())
            items << QString::fromUtf8((poster->GetTitle() + std::string(" - ") + poster->GetDisplayAuthors(m_pConfDB)).c_str());
    }

    if (items.count() == 0) {
        QMessageBox msgBox;
        msgBox.setIcon(QMessageBox::Warning);
        msgBox.setText("No additional posters available.");
        msgBox.exec();
        return;
    }

    items.sort();

    bool ok;
    QString item = QInputDialog::getItem(this, tr("Select Poster"),
                                         tr("Available posters:"), items, 0, false, &ok);
    if (ok && !item.isEmpty()) {
        for (size_t i = 0;i<m_pConfDB->works.Count();++i) {
            const Poster* poster = dynamic_cast<const Poster*>(m_pConfDB->works.GetEntryByPos(i));
            if (!poster) continue;

            if (item == QString::fromUtf8((poster->GetTitle() + std::string(" - ") + poster->GetDisplayAuthors(m_pConfDB)).c_str())) {
                p->AddPoster(*poster);
                break;
            }
        }

        UpdateSessionsUI(sIndex);
        UpdatePublicationUI();
        conferenceDataChanged();
    }
}

void MainWindow::posterSessionPostersDelElement(index_t  index) {
    index_t sIndex = m_SessionList->GetActiveIndex();
    PosterSession* p = dynamic_cast<PosterSession*>(m_pConfDB->sessions.GetEntryByIndex(sIndex));

    p->DelPoster(index);

    UpdateSessionsUI(sIndex);
    UpdatePublicationUI();
}

void MainWindow::invitedSessionTalksNewElement() {
    QStringList items;
    index_t sIndex = m_SessionList->GetActiveIndex();
    Keynote* k = dynamic_cast<Keynote*>(m_pConfDB->sessions.GetEntryByIndex(sIndex));
    assert(k);

    for (size_t i = 0;i<m_pConfDB->talks.Count();++i) {
        const InvitedTalk* talk = dynamic_cast<const InvitedTalk*>(m_pConfDB->talks.GetEntryByPos(i));
        if (!talk) continue;

        if (talk->GetReferenceCount(m_pConfDB) == 0)
            items << QString::fromUtf8((talk->GetTitle() + std::string(" - ") + talk->GetDisplaySpeakers(m_pConfDB)).c_str());
    }

    if (items.count() == 0) {
        QMessageBox msgBox;
        msgBox.setIcon(QMessageBox::Warning);
        msgBox.setText("No additional invited talks available.");
        msgBox.exec();
        return;
    }

    items.sort();

    bool ok;
    QString item = QInputDialog::getItem(this, tr("Select Talk"),
                                         tr("Available invited talks:"), items, 0, false, &ok);
    if (ok && !item.isEmpty()) {
        for (size_t i = 0;i<m_pConfDB->talks.Count();++i) {
            const InvitedTalk* talk = dynamic_cast<const InvitedTalk*>(m_pConfDB->talks.GetEntryByPos(i));
            if (!talk) continue;

            if (item == QString::fromUtf8((talk->GetTitle() + std::string(" - ") + talk->GetDisplaySpeakers(m_pConfDB)).c_str())) {
                k->AddInvitedTalk(*talk);
                break;
            }
        }

        UpdateStartEnd(sIndex);
        UpdateSessionsUI(sIndex);
        UpdateTalksUI();
        conferenceDataChanged();
    }
}

void MainWindow::invitedSessionTalksDelElement(index_t index) {
    index_t sIndex = m_SessionList->GetActiveIndex();
    Keynote* k = dynamic_cast<Keynote*>(m_pConfDB->sessions.GetEntryByIndex(sIndex));

    k->DelInvitedTalk(index);

    UpdateSessionsUI(sIndex);
    UpdateTalksUI();
}

void MainWindow::organizedSessionOrganizersNewElement() {
    QStringList items;
    index_t sIndex = m_SessionList->GetActiveIndex();
    OrganizedSession* o = dynamic_cast<OrganizedSession*>(m_pConfDB->sessions.GetEntryByIndex(sIndex));
    assert(o);

    for (size_t i = 0;i<m_pConfDB->people.Count();++i) {
        const SciPerson* pers = m_pConfDB->people.GetEntryByPos(i);
        if (std::find(o->GetOrganizers().begin(), o->GetOrganizers().end(), pers->GetIndex()) == o->GetOrganizers().end())
            items << QString::fromUtf8(pers->GetSortableDisplayName(m_pConfDB).c_str());
    }

    if (items.count() == 0) {
        QMessageBox msgBox;
        msgBox.setIcon(QMessageBox::Warning);
        msgBox.setText("No additional organizers available.");
        msgBox.exec();
        return;
    }

    items.sort();

    bool ok;
    QString item = QInputDialog::getItem(this, tr("Select Organizer"),
                                         tr("Available Organizers:"), items, 0, false, &ok);
    if (ok && !item.isEmpty()) {
        for (size_t i = 0;i<m_pConfDB->people.Count();++i) {
            const SciPerson* pers = m_pConfDB->people.GetEntryByPos(i);
            if (item == QString::fromUtf8(pers->GetSortableDisplayName(m_pConfDB).c_str())) {
                o->AddOrganizer(*pers);
                break;
            }
        }

        UpdateSessionsUI(sIndex);
        UpdatePeopleUI();
        conferenceDataChanged();
    }
}

void MainWindow::organizedSessionOrganizersDelElement(index_t index) {
    index_t sIndex = m_SessionList->GetActiveIndex();
    OrganizedSession* o = dynamic_cast<OrganizedSession*>(m_pConfDB->sessions.GetEntryByIndex(sIndex));

    o->DelOrganizer(index);

    UpdateSessionsUI(sIndex);
    UpdatePeopleUI();
}

void MainWindow::tutorialSessionTalksNewElement() {
    QStringList items;
    index_t sIndex = m_SessionList->GetActiveIndex();
    Tutorial* t = dynamic_cast<Tutorial*>(m_pConfDB->sessions.GetEntryByIndex(sIndex));
    assert(t);

    for (size_t i = 0;i<m_pConfDB->talks.Count();++i) {
        const InvitedTalk* talk = dynamic_cast<const InvitedTalk*>(m_pConfDB->talks.GetEntryByPos(i));
        if (!talk) continue;

        if (std::find(t->GetTutorialTalks().begin(), t->GetTutorialTalks().end(), talk->GetIndex()) == t->GetTutorialTalks().end())
            items << QString::fromUtf8((talk->GetTitle() + std::string(" - ") + talk->GetDisplaySpeakers(m_pConfDB)).c_str());
    }

    if (items.count() == 0) {
        QMessageBox msgBox;
        msgBox.setIcon(QMessageBox::Warning);
        msgBox.setText("No additional invited talks available.");
        msgBox.exec();
        return;
    }

    items.sort();

    bool ok;
    QString item = QInputDialog::getItem(this, tr("Select Talk"),
                                         tr("Available invited talks:"), items, 0, false, &ok);
    if (ok && !item.isEmpty()) {
        for (size_t i = 0;i<m_pConfDB->talks.Count();++i) {
            const InvitedTalk* talk = dynamic_cast<const InvitedTalk*>(m_pConfDB->talks.GetEntryByPos(i));
            if (!talk) continue;

            if (item == QString::fromUtf8((talk->GetTitle() + std::string(" - ") + talk->GetDisplaySpeakers(m_pConfDB)).c_str())) {
                t->AddTutorialTalk(*talk);
                break;
            }
        }

        UpdateStartEnd(sIndex);
        UpdateSessionsUI(sIndex);
        UpdateTalksUI();
        conferenceDataChanged();
    }
}

void MainWindow::tutorialSessionTalksDelElement(index_t index) {
    index_t sIndex = m_SessionList->GetActiveIndex();
    Tutorial* t = dynamic_cast<Tutorial*>(m_pConfDB->sessions.GetEntryByIndex(sIndex));

    t->DelTutorialTalk(index);

    UpdateSessionsUI(sIndex);
    UpdateTalksUI();
}

void MainWindow::panelSessionPanelistsNewElement() {
    QStringList items;
    index_t sIndex = m_SessionList->GetActiveIndex();
    Panel* p = dynamic_cast<Panel*>(m_pConfDB->sessions.GetEntryByIndex(sIndex));
    assert(p);

    for (size_t i = 0;i<m_pConfDB->people.Count();++i) {
        const SciPerson* pers = m_pConfDB->people.GetEntryByPos(i);
        if (std::find(p->GetPanelists().begin(), p->GetPanelists().end(), pers->GetIndex()) == p->GetPanelists().end())
            items << QString::fromUtf8(pers->GetSortableDisplayName(m_pConfDB).c_str());
    }

    if (items.count() == 0) {
        QMessageBox msgBox;
        msgBox.setIcon(QMessageBox::Warning);
        msgBox.setText("No additional panelists available.");
        msgBox.exec();
        return;
    }

    items.sort();

    bool ok;
    QString item = QInputDialog::getItem(this, tr("Select panelist"),
                                         tr("Available panelists:"), items, 0, false, &ok);
    if (ok && !item.isEmpty()) {
        for (size_t i = 0;i<m_pConfDB->people.Count();++i) {
            const SciPerson* pers = m_pConfDB->people.GetEntryByPos(i);
            if (item == QString::fromUtf8(pers->GetSortableDisplayName(m_pConfDB).c_str())) {
                p->AddPanelist(*pers);
                break;
            }
        }

        UpdateSessionsUI(sIndex);
        UpdatePeopleUI();
        conferenceDataChanged();
    }
}

void MainWindow::panelSessionPanelistsDelElement(index_t index) {
    index_t sIndex = m_SessionList->GetActiveIndex();
    Panel* p = dynamic_cast<Panel*>(m_pConfDB->sessions.GetEntryByIndex(sIndex));

    p->DelPanelist(index);

    UpdateSessionsUI(sIndex);
    UpdatePeopleUI();
}
