#include "QElemList.h"
#include "../../CrossPlatform/DemoDBGenerator.h"

#include <QtGui/QMessageBox>
#include <QtGui/QFont>
#include <QtGui/QPushButton>
#include <QtCore/QCoreApplication>
#include <algorithm>

QElemList::QElemList(QWidget *parent, bool bSort, bool bCareAboutRefs, bool bReorderWidgets)
    : QWidget(parent),
      m_iLastRow(-1),
      m_bDontEmitRowChange(false),
      m_bSort(bSort),
      m_bCareAboutRefs(bCareAboutRefs),
      m_bReorderWidgets(bReorderWidgets)
{
    setupUi();
}

QElemList::~QElemList()
{
}


void QElemList::setupUi() {
//    this->resize(800, 600);
    m_GlobalVerticalLayout = new QVBoxLayout(this);
    m_GlobalVerticalLayout->setSpacing(6);
    m_GlobalVerticalLayout->setContentsMargins(0, 0, 0, 0);
    m_GlobalVerticalLayout->setObjectName(QString::fromUtf8("m_GlobalVerticalLayout"));
    m_ElementListWidget = new QListWidget(this);
    m_ElementListWidget->setObjectName(QString::fromUtf8("m_ElementListWidget"));

    m_GlobalVerticalLayout->addWidget(m_ElementListWidget);

    m_ButtonFrame = new QFrame(this);
    m_ButtonFrame->setObjectName(QString::fromUtf8("m_ButtonFrame"));
    m_ButtonFrame->setFrameShape(QFrame::NoFrame);
    m_ButtonFrame->setFrameShadow(QFrame::Raised);
    m_ButtonFrame->setLineWidth(0);
    m_ButtonHorizontalLayout = new QHBoxLayout(m_ButtonFrame);
    m_ButtonHorizontalLayout->setSpacing(6);
    m_ButtonHorizontalLayout->setContentsMargins(0, 0, 0, 0);
    m_ButtonHorizontalLayout->setObjectName(QString::fromUtf8("m_ButtonHorizontalLayout"));

    m_AddToolButton = new QToolButton(m_ButtonFrame);
    m_AddToolButton->setObjectName(QString::fromUtf8("m_AddToolButton"));
    m_AddToolButton->setText(QString::fromUtf8("Add New Entry"));
    m_ButtonHorizontalLayout->addWidget(m_AddToolButton);

    m_DeleteToolButton = new QToolButton(m_ButtonFrame);
    m_DeleteToolButton->setObjectName(QString::fromUtf8("m_DeleteToolButton"));
    m_DeleteToolButton->setText(QString::fromUtf8("Delete Entry"));
    m_DeleteToolButton->setEnabled(false);
    m_ButtonHorizontalLayout->addWidget(m_DeleteToolButton);

    if (m_bCareAboutRefs) {
        m_DelUnrefToolButton = new QToolButton(m_ButtonFrame);
        m_DelUnrefToolButton->setObjectName(QString::fromUtf8("m_DelUnrefToolButton"));
        m_DelUnrefToolButton->setText(QString::fromUtf8("Delete Unreferenced"));
        m_DelUnrefToolButton->setEnabled(false);
        m_ButtonHorizontalLayout->addWidget(m_DelUnrefToolButton);
        QObject::connect(m_DelUnrefToolButton, SIGNAL(clicked()), this, SLOT(delUnrefButtonPressed()));
    } else {
        m_DelUnrefToolButton = NULL;
    }

    if (m_bReorderWidgets) {
        m_UpToolButton = new QToolButton(m_ButtonFrame);
        m_UpToolButton->setObjectName(QString::fromUtf8("m_UpToolButton"));
        m_UpToolButton->setText(QString::fromUtf8("Up"));
        m_UpToolButton->setEnabled(false);
        m_ButtonHorizontalLayout->addWidget(m_UpToolButton);

        m_DownToolButton = new QToolButton(m_ButtonFrame);
        m_DownToolButton->setObjectName(QString::fromUtf8("m_DownToolButton"));
        m_DownToolButton->setText(QString::fromUtf8("Down"));
        m_DownToolButton->setEnabled(false);
        m_ButtonHorizontalLayout->addWidget(m_DownToolButton);

        QObject::connect(m_UpToolButton, SIGNAL(clicked()), this, SLOT(upButtonPressed()));
        QObject::connect(m_DownToolButton, SIGNAL(clicked()), this, SLOT(downButtonPressed()));
    } else {
        m_UpToolButton = NULL;
        m_DownToolButton = NULL;
    }

    m_CountLabel = new QLabel(m_ButtonFrame);
    m_CountLabel->setObjectName(QString::fromUtf8("m_CountLabel"));
    m_CountLabel->setText("0/0");
    m_ButtonHorizontalLayout->addWidget(m_CountLabel);

    m_ButtonHorizontalSpacer = new QSpacerItem(20, 40, QSizePolicy::Expanding, QSizePolicy::Minimum);
    m_ButtonHorizontalLayout->addItem(m_ButtonHorizontalSpacer);
    m_GlobalVerticalLayout->addWidget(m_ButtonFrame);

    QObject::connect(m_AddToolButton, SIGNAL(clicked()), this, SLOT(addButtonPressed()));
    QObject::connect(m_DeleteToolButton, SIGNAL(clicked()), this, SLOT(delButtonPressed()));
    QObject::connect(m_ElementListWidget, SIGNAL(currentRowChanged(int)), this, SLOT(currentRowChanged(int)));
}

static bool comp (const ElemListEntry& i, const ElemListEntry& j) {
    return (i.m_title<j.m_title);
}

void QElemList::ElemUpdate(const std::vector< ElemListEntry >& list, index_t newIndex) {
    m_list = list;
    if (m_bSort) std::stable_sort(m_list.begin(), m_list.end(), comp);
    UpdateListView();
    SetActiveIndex(newIndex);
}

void QElemList::upButtonPressed(){
    emit upElement(GetActiveIndex());
}

void QElemList::downButtonPressed(){
    emit downElement(GetActiveIndex());
}

void QElemList::addButtonPressed(){
    emit newElement();
}

void QElemList::delButtonPressed() {
    if (m_ElementListWidget->currentRow() < 0 || m_ElementListWidget->currentRow() >= int(m_list.size())) return;

    unsigned int iRefCount = m_list[m_ElementListWidget->currentRow()].m_refcount;
    if (iRefCount == 0) {
        emit delElement(GetActiveIndex());
    } else {
        QMessageBox msgBox;

        if (iRefCount > 1)
            msgBox.setText(tr("Entry cannot be deleted, it is referenced %1 times.").arg(iRefCount));
        else
            msgBox.setText("Entry cannot be deleted, it is referenced once.");

        if (m_list.size() > 1 && receivers(SIGNAL(replaceElement(index_t))) > 0) {
            msgBox.setText(msgBox.text() + tr(" Do you want to replace it by another reference?"));
            msgBox.setIcon(QMessageBox::Question);
            msgBox.addButton(tr("Replace"), QMessageBox::ActionRole);
            QPushButton *abortButton = msgBox.addButton(QMessageBox::Abort);
            msgBox.exec();
            if (msgBox.clickedButton() == abortButton) return;
            emit replaceElement(GetActiveIndex());
        }  else {
            msgBox.setIcon(QMessageBox::Warning);
            msgBox.exec();
        }
    }
}

void QElemList::delUnrefButtonPressed() {
    size_t index = 0;
    while (index < m_list.size()) {
        if (m_list[index].m_refcount == 0) {
          emit delElement(m_list[index].m_index);
        } else {
            ++index;
        }
    }
}

void QElemList::currentRowChanged(int iRow) {
    if (m_bDontEmitRowChange) return;

    index_t index = GetActiveIndex(iRow);
    if (int(index) != m_iLastRow) {
        emit activeElement(index);
        m_iLastRow = index;
    }

    m_DeleteToolButton->setEnabled(iRow >= 0);
    if (m_bCareAboutRefs) m_DelUnrefToolButton->setEnabled(iRow >= 0);
    if (m_bReorderWidgets) {
        m_UpToolButton->setEnabled(iRow > 0);
        m_DownToolButton->setEnabled(iRow+1 < m_ElementListWidget->count());
    }

}

index_t QElemList::GetActiveIndex() {
    return GetActiveIndex(m_ElementListWidget->currentRow());
}

index_t QElemList::GetActiveIndex(int iRow) {
    if (iRow >= 0 && iRow < int(m_list.size()))
        return m_list[iRow].m_index;
    else
        return 0;
}

void QElemList::SetActiveIndex(index_t newIndex) {
    int iRow = m_ElementListWidget->count() ? 0 : -1;
    for (size_t i = 0;i<m_list.size();++i) {
        if (m_list[i].m_index == newIndex) {
            iRow = i;
            break;
        }
    }
    m_ElementListWidget->setCurrentRow(iRow);
}

void QElemList::UpdateListView() {
    m_bDontEmitRowChange = !m_list.empty();
    m_ElementListWidget->clear();
    m_bDontEmitRowChange = false;

    size_t unrefed = 0;
    for (size_t i = 0;i<m_list.size();++i) {
       m_ElementListWidget->addItem(QString::fromUtf8(m_list[i].m_title.c_str()));

       if (m_bCareAboutRefs) {
           if (m_list[i].m_refcount > 0) {
               QFont f = m_ElementListWidget->item(i)->font();
               f.setBold(true);
               m_ElementListWidget->item(i)->setFont(f);

               QBrush b = m_ElementListWidget->item(i)->foreground();
               b.setColor(Qt::blue);
               m_ElementListWidget->item(i)->setForeground(b);
           } else {
               ++unrefed;
           }
       }
    }

    if (m_bCareAboutRefs)
        m_CountLabel->setText(tr("%1/%2").arg(unrefed).arg(m_list.size()));
    else
        m_CountLabel->setText(tr("%1").arg(m_list.size()));
}

void QElemList::Clear() {
    std::vector< ElemListEntry > list;
    ElemUpdate(list, INDEX_EMPTY);
}
