#include "mainwindow.h"

#include <QtGui/QMessageBox>
#include <QtGui/QFileDialog>
#include <QtGui/QInputDialog>
#include <QtGui/QDesktopServices>

#include <QtCore/QUrl>
#include <QtCore/QFileInfo>
#include <QtCore/QFile>
#include <QtCore/QDir>

#include "QImportDialog.h"

void MainWindow::setupBuildingsTab(){
    m_tabBuildings = new QWidget();
    m_tabBuildings->setObjectName(QString::fromUtf8("m_tabBuildings"));
    m_tabBuildings->setWindowTitle(QString::fromUtf8("Buildings"));
    m_globalTabs->addTab(m_tabBuildings, m_tabBuildings->windowTitle());

    m_BuildingGlobalLayout = new QHBoxLayout(m_tabBuildings);
    m_BuildingGlobalLayout->setSpacing(6);
    m_BuildingGlobalLayout->setContentsMargins(0, 0, 0, 0);
    m_BuildingGlobalLayout->setObjectName(QString::fromUtf8("m_BuildingGlobalLayout"));
    m_BuildingSplitter = new QSplitter();
    m_BuildingGlobalLayout->addWidget(m_BuildingSplitter);

    m_BuildingListFrame = new QFrame(m_tabBuildings);
    m_BuildingListFrame->setObjectName(QString::fromUtf8("m_BuildingListFrame"));
    m_BuildingListFrame->setFrameShape(QFrame::NoFrame);
    m_BuildingListFrame->setLineWidth(0);
    m_gridLayoutBuildingList = new QGridLayout(m_BuildingListFrame);
    m_gridLayoutBuildingList->setSpacing(6);
    m_gridLayoutBuildingList->setContentsMargins(0, 0, 0, 0);
    m_gridLayoutBuildingList->setObjectName(QString::fromUtf8("m_gridLayoutBuildingList"));

    m_BuildingImportBtn = new QPushButton(m_tabBuildings);
    m_BuildingImportBtn->setText("Import Buildings from CSV-File");
    m_gridLayoutBuildingList->addWidget(m_BuildingImportBtn, 0, 0, 1, 1);

    m_BuildingList = new QElemList(m_tabBuildings, true);
    m_BuildingList->setObjectName(QString::fromUtf8("m_BuildingList"));
    m_gridLayoutBuildingList->addWidget(m_BuildingList, 1, 0, 1, 1);

    m_BuildingEditFrame = new QFrame(m_tabBuildings);
    m_BuildingEditFrame->setObjectName(QString::fromUtf8("m_BuildingEditFrame"));
    m_BuildingEditFrame->setFrameShape(QFrame::NoFrame);
    m_BuildingEditFrame->setFrameShadow(QFrame::Raised);
    m_BuildingEditFrame->setLineWidth(0);
    m_gridLayoutBuilding = new QGridLayout(m_BuildingEditFrame);
    m_gridLayoutBuilding->setSpacing(6);
    m_gridLayoutBuilding->setContentsMargins(0, 0, 0, 0);
    m_gridLayoutBuilding->setObjectName(QString::fromUtf8("m_gridLayoutBuilding"));

    m_BuildingNameLabel = new QLabel(m_BuildingEditFrame);
    m_BuildingNameLabel->setObjectName(QString::fromUtf8("m_BuildingNameLabel"));
    m_BuildingNameLabel->setText(QString::fromUtf8("Name"));
    m_gridLayoutBuilding->addWidget(m_BuildingNameLabel, 0, 0, 1, 1);
    m_BuildingNameEdit = new QLineEdit(m_BuildingEditFrame);
    m_BuildingNameEdit->setObjectName(QString::fromUtf8("m_BuildingNameEdit"));
    m_gridLayoutBuilding->addWidget(m_BuildingNameEdit, 0, 1, 1, 4);

    m_BuildingLatitudeLabel = new QLabel(m_BuildingEditFrame);
    m_BuildingLatitudeLabel->setObjectName(QString::fromUtf8("m_BuildingLatitudeLabel"));
    m_BuildingLatitudeLabel->setText(QString::fromUtf8("Latitude"));
    m_gridLayoutBuilding->addWidget(m_BuildingLatitudeLabel, 1, 0, 1, 1);
    m_BuildingLatitudeEdit = new QDoubleSpinBox(m_BuildingEditFrame);
    m_BuildingLatitudeEdit->setObjectName(QString::fromUtf8("m_BuildingLatitudeEdit"));
    m_BuildingLatitudeEdit->setRange(-90,90);
    m_BuildingLatitudeEdit->setDecimals(8);
    m_gridLayoutBuilding->addWidget(m_BuildingLatitudeEdit, 1, 1, 1, 1);

    m_BuildingLongitudeLabel = new QLabel(m_BuildingEditFrame);
    m_BuildingLongitudeLabel->setObjectName(QString::fromUtf8("m_BuildingLongitudeLabel"));
    m_BuildingLongitudeLabel->setText(QString::fromUtf8("Longitude"));
    m_BuildingLongitudeLabel->setAlignment(Qt::AlignLeading|Qt::AlignRight|Qt::AlignCenter);
    m_gridLayoutBuilding->addWidget(m_BuildingLongitudeLabel, 1, 2, 1, 1);
    m_BuildingLongitudeEdit = new QDoubleSpinBox(m_BuildingEditFrame);
    m_BuildingLongitudeEdit->setObjectName(QString::fromUtf8("m_BuildingLongitudeEdit"));
    m_BuildingLongitudeEdit->setRange(-180,180);
    m_BuildingLongitudeEdit->setDecimals(8);
    m_gridLayoutBuilding->addWidget(m_BuildingLongitudeEdit, 1, 3, 1, 1);

    m_BuildingShowOnMap = new QToolButton(m_BuildingEditFrame);
    m_BuildingShowOnMap->setObjectName(QString::fromUtf8("m_BuildingShowOnMap"));
    m_BuildingShowOnMap->setText(QString::fromUtf8("..."));
    m_gridLayoutBuilding->addWidget(m_BuildingShowOnMap, 1, 4, 1, 1);
    QObject::connect(m_BuildingShowOnMap, SIGNAL(clicked()), this, SLOT(buildingShowOnGoogleMaps()));

    m_BuildingFloorplansLabel = new QLabel(m_BuildingEditFrame);
    m_BuildingFloorplansLabel->setObjectName(QString::fromUtf8("m_BuildingFloorplansLabel"));
    m_BuildingFloorplansLabel->setText(QString::fromUtf8("Floorplans"));
    m_gridLayoutBuilding->addWidget(m_BuildingFloorplansLabel, 2, 0, 1, 1);

    m_BuildingFloorplansList = new QElemList(m_BuildingEditFrame, false, false);
    m_BuildingFloorplansList->setObjectName(QString::fromUtf8("m_BuildingFloorplansList"));
    m_gridLayoutBuilding->addWidget(m_BuildingFloorplansList, 3, 0, 1, 5);

    m_BuildingEditSpacer = new QSpacerItem(20, 40, QSizePolicy::Minimum, QSizePolicy::Expanding);
    m_gridLayoutBuilding->addItem(m_BuildingEditSpacer, 4, 0, 1, 1);

    m_BuildingSplitter->addWidget(m_BuildingListFrame);
    m_BuildingSplitter->addWidget(m_BuildingEditFrame);

    QObject::connect(m_BuildingNameEdit, SIGNAL(textEdited(QString)), this, SLOT(buildingsDataChanged()));
    QObject::connect(m_BuildingNameEdit, SIGNAL(textEdited(QString)), this, SLOT(conferenceDataChanged()));
    QObject::connect(m_BuildingLatitudeEdit, SIGNAL(valueChanged(double)), this, SLOT(buildingsDataChanged()));
    QObject::connect(m_BuildingLatitudeEdit, SIGNAL(valueChanged(double)), this, SLOT(conferenceDataChanged()));
    QObject::connect(m_BuildingLongitudeEdit, SIGNAL(valueChanged(double)), this, SLOT(buildingsDataChanged()));
    QObject::connect(m_BuildingLongitudeEdit, SIGNAL(valueChanged(double)), this, SLOT(conferenceDataChanged()));

    QObject::connect(m_BuildingList, SIGNAL(newElement()), this, SLOT(buildingsNewElement()));
    QObject::connect(m_BuildingList, SIGNAL(newElement()), this, SLOT(conferenceDataChanged()));
    QObject::connect(m_BuildingList, SIGNAL(delElement(index_t)), this, SLOT(buildingsDelElement(index_t)));
    QObject::connect(m_BuildingList, SIGNAL(delElement(index_t)), this, SLOT(conferenceDataChanged()));
    QObject::connect(m_BuildingList, SIGNAL(activeElement(index_t)), this, SLOT(buildingsActiveElement(index_t)));

    QObject::connect(m_BuildingFloorplansList, SIGNAL(newElement()), this, SLOT(buildingsFloorplansNewElement()));
    QObject::connect(m_BuildingFloorplansList, SIGNAL(delElement(index_t)), this, SLOT(buildingsFloorplansDelElement(index_t)));

    QObject::connect(m_BuildingImportBtn, SIGNAL(clicked()), this, SLOT(buidlingImportCSV()));

    buildingsActiveElement(INDEX_EMPTY);
}

void MainWindow::buidlingImportCSV() {
    if (!SaveCheck("importing buildings from CSV file")) return;

    std::vector<std::string> properties;
    properties.push_back("Name");
    properties.push_back("GPS latitude");
    properties.push_back("GPS longitude");

    QImportDialog d(properties, this);
    if (QDialog::Accepted != d.exec()) return;

    std::string filename = d.GetFilename();
    if (!filename.empty()) {
        std::vector<int> choices = d.GetChoices();
        CSFParser csvData(filename,d.GetOffset(),d.RemoveWhitespaces());
        csvData.SetRecordSizeToMax();

        for (size_t i = 0;i<csvData.RecordCount();i++) {
            std::vector< std::string > csvRecord = csvData.GetRecord(i);
            Building b(choices[0]>= 0 ? csvRecord[choices[0]] : "",
                       std::make_pair(
                           choices[1]>= 0 ? atof(csvRecord[choices[1]].c_str()) : 0,
                           choices[2]>= 0 ? atof(csvRecord[choices[2]].c_str()) : 0
                                     )
                       );
            m_pConfDB->buildings.AddUniqueEntry(b);
        }

        conferenceDataChanged();
        UpdateUI();
    }
}

void MainWindow::UpdateBuildingsUI(index_t newIndex) {
    std::vector< ElemListEntry > bList;
    for (size_t i = 0;i<m_pConfDB->buildings.Count();++i) {
        const Building* b = m_pConfDB->buildings.GetEntryByPos(i);
        std::stringstream ss;
        ss << b->GetName().c_str()  << " (" << b->GetPos().first << ", " << b->GetPos().second << ")";

        bList.push_back( ElemListEntry(ss.str(),
                                         b->GetIndex(),
                                         b->GetReferenceCount(m_pConfDB))  );
    }
    m_BuildingList->ElemUpdate(bList, newIndex);
    UpdateBuildingsFloorplans(m_BuildingList->GetActiveIndex());
}

void MainWindow::UpdateBuildingsFloorplans() {
    index_t i = m_BuildingList->GetActiveIndex();
    if (!i) return;
    UpdateBuildingsFloorplans(i);
}

void MainWindow::UpdateBuildingsFloorplans(index_t index) {
    Building* b = m_pConfDB->buildings.GetEntryByIndex(index);

    std::vector< ElemListEntry > floorplanList;
    if (b) {
        for (size_t i = 0;i<b->GetFloorplans().size();++i) {
            Floorplan fp = b->GetFloorplans()[i];
            floorplanList.push_back( ElemListEntry(fp.m_desc +
                                                   std::string(" (") +
                                                   fp.m_filename + std::string(")"),
                                                   i,
                                                   b->GetFPReferenceCount(m_pConfDB,i)));
        }
    }
    m_BuildingFloorplansList->ElemUpdate(floorplanList, INDEX_EMPTY);
}


// Slots

void MainWindow::buildingsDataChanged(){
    if (m_bEditBlockSignals) return;
    index_t index = m_BuildingList->GetActiveIndex();
    Building* b = m_pConfDB->buildings.GetEntryByIndex(index);

    b->SetName(std::string(m_BuildingNameEdit->text().toUtf8()));
    b->SetPos(std::make_pair(m_BuildingLatitudeEdit->value(), m_BuildingLongitudeEdit->value()));

    UpdateBuildingsUI(index);
    RoomsFillCBoxes();
}

void MainWindow::buildingsNewElement(){
    Building newBuilding("New", std::make_pair(0,0));
    index_t index = m_pConfDB->buildings.AddEntry(newBuilding);
    UpdateBuildingsUI(index);
}

void MainWindow::buildingsDelElement(index_t index) {
    const Building* b = m_pConfDB->buildings.GetEntryByIndex(index);
    assert(b);
    assert(b->GetReferenceCount(m_pConfDB) == 0);

    m_pConfDB->buildings.RemoveEntryByIndex(index);
    UpdateBuildingsUI(INDEX_EMPTY);
}

void MainWindow::buildingsActiveElement(index_t index) {
    const Building* b = m_pConfDB->buildings.GetEntryByIndex(index);
    bool bo = StartEditBlocker();
    if (b) {
        m_BuildingNameEdit->setText(QString::fromUtf8(b->GetName().c_str()));
        m_BuildingLatitudeEdit->setValue(b->GetPos().first);
        m_BuildingLongitudeEdit->setValue(b->GetPos().second);

        UpdateBuildingsFloorplans(index);

        m_BuildingNameEdit->setEnabled(true);
        m_BuildingLatitudeEdit->setEnabled(true);
        m_BuildingLongitudeEdit->setEnabled(true);
        m_BuildingFloorplansList->setEnabled(true);
    } else {
        m_BuildingNameEdit->setText("");
        m_BuildingLatitudeEdit->setValue(0);
        m_BuildingLongitudeEdit->setValue(0);
        m_BuildingFloorplansList->Clear();
        m_BuildingNameEdit->setEnabled(false);
        m_BuildingLatitudeEdit->setEnabled(false);
        m_BuildingLongitudeEdit->setEnabled(false);
        m_BuildingFloorplansList->setEnabled(false);
    }
    EndEditBlocker(bo);
}

void MainWindow::buildingsFloorplansNewElement() {
    if (m_strCurrentFile.isEmpty()) {
        QMessageBox::StandardButton reply;
        reply = QMessageBox::question(this, tr("Adding Floorplan"),
                                      tr("You need to save the conference database at least once before you can add floorplans. Do you want to save the database now?"),
                                        QMessageBox::Save | QMessageBox::Cancel, QMessageBox::Save);
        if (reply == QMessageBox::Save) {
            saveDB();
            if (m_strCurrentFile.count() == 0) return;
        } else if (reply == QMessageBox::Cancel) {
            return;
        }
    }

    QString imageFile = QFileDialog::getOpenFileName(this,"Floorplan Wizzard step 1/3: Select a floorplan image", QString::null, "Images (*.jpg *.png)");
    if (imageFile.isEmpty()) return;

    bool ok;
    QString desc = QInputDialog::getText(this, tr("Floorplan Wizzard step 2/3: Floor decription"),
                                         tr("Name of the floor (e.g. Ground level):"), QLineEdit::Normal, "", &ok);
    if (!ok || desc.isEmpty()) return;


    int width = QInputDialog::getInt(this, tr("Floorplan Wizzard step 3/3: Pin Size"),
                                                tr("What width (in pixel) should the location pin have:"), 100, 1, 100000, 1, &ok);
    if (!ok) return;

    QFileInfo source(imageFile);

    QFileInfo dbFile(m_strCurrentFile);
    QFileInfo target(tr("%1/%2").arg(dbFile.dir().canonicalPath()).arg(source.fileName()));

    bool copyOK = true;

    if (source.filePath() != target.filePath()) {
        if (target.exists()) {
            QMessageBox msgBox(QMessageBox::Question,tr("Image Collision"), tr("An image with the same name already exists in the database directroy"));
            msgBox.setInformativeText(tr("Do you want to override the file--thus possibly affecting other floorplans--or use the existing image for this floorplan, too?"));
            QPushButton *overrideButton = msgBox.addButton(tr("Override"), QMessageBox::ActionRole);
            msgBox.addButton(tr("Reuse Existing"), QMessageBox::ActionRole);
            QPushButton *cancelButton = msgBox.addButton(QMessageBox::Cancel);
            msgBox.setDefaultButton(cancelButton);
            msgBox.exec();
            if (msgBox.clickedButton() == cancelButton) return;

            if (msgBox.clickedButton() == overrideButton)
                copyOK = QFile::copy(source.filePath(), target.filePath());
        } else {
            copyOK = QFile::copy(source.filePath(), target.filePath());
        }
    }


    if (!copyOK) {
        QMessageBox msgBox;
        msgBox.setIcon(QMessageBox::Critical);
        msgBox.setText(tr("Unable to copy image to database directory (%1 to %2)!").arg(source.filePath()).arg(target.filePath()));
        msgBox.exec();
        return;
    }

    Floorplan f(std::string(desc.toUtf8()), std::string(source.fileName().toUtf8()), float(width)/float(286));  // 286 is the width of the needle icon
    index_t index = m_BuildingList->GetActiveIndex();
    Building* b = m_pConfDB->buildings.GetEntryByIndex(index);
    b->AddFloorplan(f);
    UpdateBuildingsFloorplans(index);
    RoomsFillCBoxes();
    conferenceDataChanged();
}

void MainWindow::buildingsFloorplansDelElement(index_t index) {
    index_t bIndex = m_BuildingList->GetActiveIndex();
    Building* b = m_pConfDB->buildings.GetEntryByIndex(bIndex);

    // check if the image exists
    QFileInfo dbFile(m_strCurrentFile);
    QFileInfo imageFile(tr("%1/%2").arg(dbFile.dir().canonicalPath()).arg(b->GetFloorplans()[index].m_filename.c_str()));
    bool bExists = imageFile.exists();

    // if the image exists check if it's used elsewhere in the buildings
    bool bFound = false;
    if (bExists) {
        for (size_t i = 0;i<m_pConfDB->buildings.Count();++i) {
            const Building* bb = m_pConfDB->buildings.GetEntryByPos(i);
            for (size_t j = 0;j<bb->GetFloorplans().size();++j) {
                if (bb->GetIndex()==bIndex && j == index) continue; // skip ourself

                if (m_pConfDB->buildings.GetEntryByPos(i)->GetFloorplans()[j].m_filename == b->GetFloorplans()[index].m_filename) {
                    bFound = true;
                    break;
                }
            }
            if (bFound) break;
        }
    }

    if (bExists && !bFound) {
        QMessageBox::StandardButton reply;
        reply = QMessageBox::question(this, tr("Deleting Floorplan"),
                                      tr("The image file \"%1\" is only associated with the current floorplan and not used by any other. Do you also want to delete this image from disk?").arg(b->GetFloorplans()[index].m_filename.c_str()),
                                        QMessageBox::Yes | QMessageBox::No | QMessageBox::Cancel);
        if (reply == QMessageBox::Yes) {
           QFile::remove(imageFile.filePath());
        } else if (reply == QMessageBox::Cancel) {
            return;
        }
    }

    b->DelFloorplan(index, m_pConfDB);
    UpdateBuildingsFloorplans(bIndex);
    RoomsFillCBoxes();
    conferenceDataChanged();
}

void MainWindow::buildingShowOnGoogleMaps() {
    const Building* b = m_pConfDB->buildings.GetEntryByIndex(m_BuildingList->GetActiveIndex());
    if (!b) return;
    QUrl url(tr("http://maps.google.com/maps?q=%1,+%2+(%3)&iwloc=A&hl=en").arg(b->GetPos().first).arg(b->GetPos().second).arg(b->GetName().c_str()));
    QDesktopServices::openUrl ( url );
}
