#include "Talk.h"
#include "ConferenceDB.h"
#include "StringList.h"

Talk::Talk(const std::string& strTitle, 
           const int iStartTime,
           const int iEndTime) :
Indexable(),
m_strTitle(strTitle),
m_iStartTime(iStartTime),
m_iEndTime(iEndTime)
{
}

Talk::Talk(const std::string& strTitle, 
           const std::vector<index_t>& vSpeakers,
           const int iStartTime,
           const int iEndTime) :
Indexable(),
m_strTitle(strTitle),
m_vSpeakers(vSpeakers),
m_iStartTime(iStartTime),
m_iEndTime(iEndTime)
{
}

Talk::Talk(std::istream& stream) :
Indexable()
{
  m_bDeserializationSuccess = Deserialize(stream);
}

void Talk::SerializeValues(std::string& serializationString) const {
  Indexable::SerializeValues(serializationString);
  AddSerialization(serializationString, m_strTitle);
  AddVectorSerialization(serializationString, m_vSpeakers);  
  AddSerialization(serializationString, m_iStartTime);
  AddSerialization(serializationString, m_iEndTime);
}

bool Talk::DeserializeValues(std::istream& stream) {
  if (!Indexable::DeserializeValues(stream)) return false;
  if (!DeserializeValue(stream, m_strTitle)) return false;
  if (!DeserializeVector(stream, m_vSpeakers)) return false;
  if (!DeserializeValue(stream, m_iStartTime)) return false;
  return DeserializeValue(stream, m_iEndTime);
}

const Session* Talk::FindContainingSession(const ConferenceDB* confDB) const {
  for (size_t i = 0;i<confDB->sessions.Count();++i) {
    const Session* potentialSession = confDB->sessions.GetEntryByPos(i);
    if (potentialSession && potentialSession->Contains(this)) return potentialSession;
  }
  return NULL;
}

std::string Talk::GetDisplaySpeakers(const ConferenceDB* confDB) const {
  std::string result;
  for (size_t i = 0;i<m_vSpeakers.size();++i) {
    const SciPerson* p = confDB->people.GetEntryByIndex(m_vSpeakers[i]);
    if (p) {
      if (result.empty()) 
        result = p->GetDisplayName();
      else {
        if (i == m_vSpeakers.size()-1)
          result += std::string(", ") + StringList::strAnd + std::string(" ") + p->GetDisplayName();
        else
          result += std::string(", ") + p->GetDisplayName();
      }
    }
  }
  return result;
}

void Talk::DelSpeaker(const index_t speakerIndex) {
    for (size_t i = 0;i<m_vSpeakers.size();++i) {
      if (m_vSpeakers[i] == speakerIndex)
        m_vSpeakers.erase(m_vSpeakers.begin()+i);
    }
}

void Talk::SpeakerUp(const index_t speakerIndex) {
    for (size_t i = 0;i<m_vSpeakers.size();++i) {
      if (m_vSpeakers[i] == speakerIndex && i > 0) {
        std::swap(m_vSpeakers[i], m_vSpeakers[i-1]);
        return;
      }
    }
}

void Talk::SpeakerDown(const index_t speakerIndex) {
    for (size_t i = 0;i<m_vSpeakers.size();++i) {
      if (m_vSpeakers[i] == speakerIndex && i+1 < m_vSpeakers.size()) {
        std::swap(m_vSpeakers[i], m_vSpeakers[i+1]);
        return;
      }
    }
}

unsigned int Talk::ReplaceSpeaker(const SciPerson& person, const SciPerson& newPerson) {
    unsigned int refs = 0;
    index_t personIndex = person.GetIndex();
    for (size_t i = 0;i<m_vSpeakers.size();++i) {
      if (m_vSpeakers[i] == personIndex) {
        m_vSpeakers[i] = newPerson.GetIndex();
        ++refs;
      }
    }
    return refs;
}

bool Talk::Validate(const ConferenceDB* confDB) const {
  for (size_t i = 0;i<m_vSpeakers.size();++i) {
    if (! confDB->people.GetEntryByIndex(m_vSpeakers[i]) )
        return false;
  }  
  return true;
}

unsigned int Talk::GetReferenceCount(const ConferenceDB* confDB) const {
  unsigned int refs = 0;
  for (size_t i = 0;i<confDB->sessions.Count();++i) {
    const Session* potentialSession = confDB->sessions.GetEntryByPos(i);
    if (potentialSession->Contains(this)) refs++;
  }
  return refs;
}

