#include <algorithm>
#include "Session.h"
#include "ConferenceDB.h"

Session::Session(const std::string& strTitle,
                 const int iStartTime,
                 const int iEndTime,
                 const std::string& strDescription) :
Indexable(),
m_strTitle(strTitle),
m_strURL(""),
m_iStartTime(iStartTime),
m_iEndTime(iEndTime),
m_SessionChair(INDEX_EMPTY),
m_bChairAcceptsMail(false),
m_strDescription(strDescription)
{
}


Session::Session(const std::string& strTitle, 
                 const int iStartTime,
                 const int iEndTime,
                 const SciPerson& sessionChair,
                 const bool bChairAcceptsMail,
                 const std::string& strDescription) :
Indexable(),
m_strTitle(strTitle),
m_strURL(""),
m_iStartTime(iStartTime),
m_iEndTime(iEndTime),
m_SessionChair(sessionChair.GetIndex()),
m_bChairAcceptsMail(bChairAcceptsMail),
m_strDescription(strDescription)
{
}

Session::Session(const std::string& strTitle, 
                 const int iStartTime,
                 const int iEndTime,
                 const SciPerson& sessionChair,
                 const bool bChairAcceptsMail,
                 const std::string& strDescription,
                 const std::vector<index_t>& vRooms) :
Indexable(),
m_strTitle(strTitle),
m_strURL(""),
m_vRooms(vRooms),
m_iStartTime(iStartTime),
m_iEndTime(iEndTime),
m_SessionChair(sessionChair.GetIndex()),
m_bChairAcceptsMail(bChairAcceptsMail),
m_strDescription(strDescription)
{
}

Session::Session(std::istream& stream) :
Indexable()
{
  m_bDeserializationSuccess = Deserialize(stream);
}

void Session::SerializeValues(std::string& serializationString) const {
  Indexable::SerializeValues(serializationString);
  AddSerialization(serializationString, m_strTitle);
  AddSerialization(serializationString, m_strURL);
  AddSerialization(serializationString, m_iStartTime);
  AddSerialization(serializationString, m_iEndTime);
  AddSerialization(serializationString, m_SessionChair);
  AddSerialization(serializationString, m_bChairAcceptsMail);
  AddSerialization(serializationString, m_strDescription);
  AddVectorSerialization(serializationString, m_vRooms);
}

bool Session::DeserializeValues(std::istream& stream) {
  if (!Indexable::DeserializeValues(stream)) return false;
  if (!DeserializeValue(stream, m_strTitle)) return false;
  if (!DeserializeValue(stream, m_strURL)) return false;
  if (!DeserializeValue(stream, m_iStartTime)) return false;
  if (!DeserializeValue(stream, m_iEndTime)) return false;
  if (!DeserializeValue(stream, m_SessionChair)) return false;
  if (!DeserializeValue(stream, m_bChairAcceptsMail)) return false;
  if (!DeserializeValue(stream, m_strDescription)) return false;  
  if (!DeserializeVector(stream, m_vRooms)) return false;
  return true;
}

void Session::DelRoom(index_t index) {
    for (size_t i = 0;i<m_vRooms.size();++i) {
      if (m_vRooms[i] == index)
        m_vRooms.erase(m_vRooms.begin()+i);
    }
}

const Track* Session::FindContainingTrack(const ConferenceDB* confDB) const {
  for (size_t i = 0;i<confDB->tracks.Count();++i) {
    const Track* potentialTrack = confDB->tracks.GetEntryByPos(i);    
    const std::vector<index_t>& sessions = potentialTrack->GetSessions();    
    if (std::find(sessions.begin(), sessions.end(), m_iIndex) != sessions.end() ) {
      return potentialTrack;
    }
  }
  return NULL;
}

std::string Session::GetRoomsDisplayString(const ConferenceDB* confDB) const {
  std::string result;
  for (size_t i = 0;i<m_vRooms.size();++i) {
    const Room* r = confDB->rooms.GetEntryByIndex(m_vRooms[i]);
    if (r) {
      if (result.empty()) 
        result = r->GetDisplayName(confDB);
      else {
        if (i == m_vRooms.size()-1)
          result += std::string(", ") + StringList::strAnd + std::string(" ") + r->GetDisplayName(confDB);
        else
          result += std::string(", ") + r->GetDisplayName(confDB);
      }
    }
  }
  return result;
}

bool Session::Validate(const ConferenceDB* confDB) const {
  if (m_SessionChair != INDEX_EMPTY && !confDB->people.GetEntryByIndex(m_SessionChair) ) return false;    
  
  for (size_t i = 0;i<m_vRooms.size();++i) {
    if (! confDB->rooms.GetEntryByIndex(m_vRooms[i]) ) return false;
  }  
  return true;
}

unsigned int Session::GetReferenceCount(const ConferenceDB* confDB) const {
  unsigned int refs = 0;
  for (size_t i = 0;i<confDB->tracks.Count();++i) {
    const Track* potentialTrack = confDB->tracks.GetEntryByPos(i);    
    const std::vector<index_t>& sessions = potentialTrack->GetSessions();    
    if (std::find(sessions.begin(), sessions.end(), m_iIndex) != sessions.end() ) {
      refs++;
    }
  }
  return refs;
}
