#ifndef SERIALIZABLE_H
#define SERIALIZABLE_H

#include <iostream>
#include <fstream>
#include <sstream>
#include <vector>

namespace {
  template <typename T> std::string ToString(const T& aValue)
  {
    std::stringstream ss;
    ss << aValue;
    return ss.str();
  }

  template <> std::string ToString<std::string>(const std::string& aValue)
  {
    return aValue;
  }

  template <typename T> bool FromString(T& t, const std::string& s,
                                        std::ios_base& (*f)(std::ios_base&) = std::dec)
  {
    std::istringstream iss(s);
    return !(iss >> f >> t).fail();
  }

  template <> bool FromString(std::string& t, const std::string& s,
                              std::ios_base& (*)(std::ios_base&))
  {
    t = s;
    return true;
  }
}

class Serializable {
public:
  Serializable() : m_bDeserializationSuccess(true) {}
  
  bool SerializeFile(const std::string& strOutFile) const {
    std::ofstream stream;
    stream.open(strOutFile.c_str());
    if (!stream.is_open()) return false;    
    Serialize(stream);    
    stream.close();    
    return true;
  }

  void Serialize(std::string& serializationString) const {
    std::string s = BeginSerialization();
    SerializeValues(s);
    EndSerialization(s);

    if (serializationString == BeginSerialization()) 
      serializationString += s;
    else
      serializationString += std::string(" ") + s;
  }
  
  void Serialize(std::ostream& stream) const {
    std::string s = BeginSerialization();
    SerializeValues(s);
    EndSerialization(s, stream);    
  }
  
  bool DeserializeFile(const std::string& strInFile) {
    std::ifstream stream;
    stream.open(strInFile.c_str());
    if (!stream.is_open()) return false;    
    bool b = Deserialize(stream);    
    stream.close();    
    return b;
  }

  bool Deserialize(std::istream& stream) {
    return DeserializeValues(stream);
  }
  
  bool DidConstructionFail() {return !m_bDeserializationSuccess;}
  
  static std::string BeginSerialization() {
    return std::string("<"); // insert some spacers, this is just for the human reader and simply skipped by the parser
  }
  
  static void EndSerialization(std::string& serializationString) {
    serializationString += std::string(">"); // insert some spacers, this is just for the human reader and simply skipped by the parser
  }

  
  static void EndSerialization(std::string& serializationString, std::ostream& serializationStream) {
    EndSerialization(serializationString);
    serializationStream << serializationString.c_str();
  }
  
  template <typename T> static void AddVectorSerialization(std::string& serializationString, const std::vector<T>& vValues) {
    AddSerialization(serializationString, vValues.size());
    
    for (size_t i = 0;i<vValues.size();i++) {
      AddSerialization(serializationString,vValues[i]);
    }
  }
  
  template <typename T> static void AddClassVectorSerialization(std::string& serializationString, const std::vector<T>& vValues) {
    AddSerialization(serializationString, vValues.size());
    
    for (size_t i = 0;i<vValues.size();i++) {
      vValues[i].Serialize(serializationString);
    }
  }
  
  template <typename T, typename F> static void AddVectorSerializationWithFactory(std::string& serializationString, const std::vector<T*>& vValues) {
    AddSerialization(serializationString, vValues.size());
    
    for (size_t i = 0;i<vValues.size();i++) {
      F::Serialize(serializationString, vValues[i]);
    }
  }
  
  template <typename T> static void AddSerialization(std::string& serializationString, const T& aValue) {
    std::string strValue =  ToString(aValue);
    
    // protect """s in string
    std::string strProtectedValue("\"");
    for (size_t i = 0;i<strValue.size();i++) {
      if (strValue[i] == '"' || strValue[i] == '\\')
        strProtectedValue += std::string("\\") + strValue[i];
      else 
        strProtectedValue += strValue[i];
    }
    strProtectedValue += std::string("\"");
    
    if (serializationString == BeginSerialization()) 
      serializationString += strProtectedValue;
    else
      serializationString += std::string(" ") + strProtectedValue;
  }
  
  template <typename T> static bool DeserializeValue(std::istream& stream, T& value) {
    char singleChar;
    
    // read file until we see a " character
    do {
      singleChar = stream.get();      
      if (!stream.good()) return false;       
    } while(singleChar !='"');
        
    // copy data until we see an unprotected " character
    std::string data;
    do {
      singleChar = stream.get();
      if (!stream.good()) return false;

      if (singleChar =='\\') {
        singleChar = stream.get();
        if (!stream.good()) return false;
        data += singleChar;
        singleChar = ' '; // make sure the loop's stoping criterion is not hit
        continue;
      } 
      
      if (singleChar !='"') data += singleChar;
    } while(singleChar !='"');
    
    return FromString(value, data);
  }
  
  
  template <typename T> static bool DeserializeVector(std::istream& stream, std::vector<T>& values) {
    size_t vSize = 0;
    if (!DeserializeValue(stream, vSize)) return false;
    
    values.clear();
    for (size_t i = 0;i<vSize;i++) {
      T elment;
      if (!DeserializeValue(stream, elment)) return false;
      values.push_back(elment);
    }
    return true;
  }
  
  template <typename T> static bool DeserializeClassVector(std::istream& stream, std::vector<T>& values) {
    size_t vSize = 0;
    if (!DeserializeValue(stream, vSize)) return false;
        
    values.clear();
    for (size_t i = 0;i<vSize;i++) {
      T elment(stream);
      if (elment.DidConstructionFail()) return false;
      values.push_back(elment);
    }
    return true;
  }
  
  template <typename T, typename F> static bool DeserializeVectorWithFactory(std::istream& stream, std::vector<T*>& values) {
    size_t vSize = 0;
    if (!DeserializeValue(stream, vSize)) return false;
    
    values.clear();
    for (size_t i = 0;i<vSize;i++) {
      T* p = F::Deserialize(stream);
      if (!p) return false;
      values.push_back(p);
    }
    return true;
  }
  
protected:
  bool m_bDeserializationSuccess;
  
  virtual void SerializeValues(std::string& serializationString) const = 0;
  virtual bool DeserializeValues(std::istream& stream) = 0;
  
};


#endif // SERIALIZABLE_H
