#include "SciWork.h"
#include "ConferenceDB.h"
#include "StringList.h"

SciWork::SciWork(const std::string& strTitle, 
                 const std::string& strAbstract, 
                 const std::string& strURL, 
                 const std::string& strPreviewURL, 
                 const std::string& strPDFURL) :
Indexable(),
m_strTitle(strTitle),
m_strAbstract(strAbstract),
m_strURL(strURL),
m_strPreviewURL(strPreviewURL),
m_strPDFURL(strPDFURL)
{
}

SciWork::SciWork(const std::string& strTitle, 
                 const std::string& strAbstract, 
                 const std::string& strURL, 
                 const std::string& strPreviewURL, 
                 const std::string& strPDFURL, 
                 const std::vector<index_t>& vAuthors) :
Indexable(),
m_strTitle(strTitle),
m_strAbstract(strAbstract),
m_strURL(strURL),
m_strPreviewURL(strPreviewURL),
m_strPDFURL(strPDFURL),
m_vAuthors(vAuthors)
{
}

SciWork::SciWork(std::istream& stream) :
Indexable()
{
  m_bDeserializationSuccess = Deserialize(stream);
}

void SciWork::SerializeValues(std::string& serializationString) const {
  Indexable::SerializeValues(serializationString);
  AddSerialization(serializationString, m_strTitle);
  AddSerialization(serializationString, m_strAbstract);
  AddVectorSerialization(serializationString, m_vAuthors);
  AddSerialization(serializationString, m_strURL);
  AddSerialization(serializationString, m_strPreviewURL);
  AddSerialization(serializationString, m_strPDFURL);  
}

bool SciWork::DeserializeValues(std::istream& stream) {
  if (!Indexable::DeserializeValues(stream)) return false;
  if (!DeserializeValue(stream, m_strTitle)) return false;
  if (!DeserializeValue(stream, m_strAbstract)) return false;
  if (!DeserializeVector(stream, m_vAuthors)) return false;
  if (!DeserializeValue(stream, m_strURL)) return false;
  if (!DeserializeValue(stream, m_strPreviewURL)) return false;
  if (!DeserializeValue(stream, m_strPDFURL)) return false;

  return true;
}

std::string SciWork::GetDisplayAuthors(const ConferenceDB* db) const {
  std::string result;
  for (size_t i = 0;i<m_vAuthors.size();++i) {
    const SciPerson* p = db->people.GetEntryByIndex(m_vAuthors[i]);
    if (p) {
      if (result.empty()) 
        result = p->GetDisplayName();
      else {
        if (i == m_vAuthors.size()-1)
          result += std::string(", ") + StringList::strAnd + std::string(" ") + p->GetDisplayName();
        else
          result += std::string(", ") + p->GetDisplayName();
      }
    }
  }
  return result;
}

bool SciWork::Validate(const ConferenceDB* confDB) const {
  for (size_t i = 0;i<m_vAuthors.size();++i) {
    if (! confDB->people.GetEntryByIndex(m_vAuthors[i]) ) return false;
  }  
  return true;
}

unsigned int SciWork::ReplaceAuthor(const SciPerson& author, const SciPerson& newAuthor) {
    unsigned int refs = 0;
    index_t authorIndex = author.GetIndex();
    for (size_t i = 0;i<m_vAuthors.size();++i) {
      if (m_vAuthors[i] == authorIndex) {
        m_vAuthors[i] = newAuthor.GetIndex();
        ++refs;
      }
    }
    return refs;
}

void SciWork::DelAuthor(const index_t authorIndex) {
  for (size_t i = 0;i<m_vAuthors.size();++i) {
    if (m_vAuthors[i] == authorIndex)
      m_vAuthors.erase(m_vAuthors.begin()+i);
  }
}

void SciWork::UpAuthor(const index_t authorIndex) {
    for (size_t i = 0;i<m_vAuthors.size();++i) {
      if (m_vAuthors[i] == authorIndex && i > 0) {
        std::swap(m_vAuthors[i], m_vAuthors[i-1]);
        return;
      }
    }
}

void SciWork::DownAuthor(const index_t authorIndex) {
    for (size_t i = 0;i<m_vAuthors.size();++i) {
      if (m_vAuthors[i] == authorIndex && i+1 < m_vAuthors.size()) {
        std::swap(m_vAuthors[i], m_vAuthors[i+1]);
        return;
      }
    }
}
