#include "SciPerson.h"
#include "ConferenceDB.h"
#include "StringList.h"

#include "Panel.h"
#include "OrganizedSession.h"

SciPerson::SciPerson(const std::string& strFirstName, 
                     const std::string& strMiddleName,
                     const std::string& strLastName,
                     const std::string& strEmail,
                     const std::string& strTwitterName,
                     const std::string& strPictureURL,
                     const std::string& strWebpageURL) : 
Indexable(),
m_strFirstName(strFirstName),
m_strMiddleName(strMiddleName),
m_strLastName(strLastName),
m_strEmail(strEmail),
m_strTwitterName(strTwitterName),
m_strPictureURL(strPictureURL),
m_strWebpageURL(strWebpageURL)
{
}

SciPerson::SciPerson(const std::string& strFirstName, 
                     const std::string& strMiddleName,
                     const std::string& strLastName,
                     const std::string& strEmail,
                     const std::string& strTwitterName,
                     const std::string& strPictureURL,
                     const std::string& strWebpageURL,
                     const std::vector<index_t>& vAffiliations) : 
Indexable(),
m_strFirstName(strFirstName),
m_strMiddleName(strMiddleName),
m_strLastName(strLastName),
m_strEmail(strEmail),
m_strTwitterName(strTwitterName),
m_strPictureURL(strPictureURL),
m_strWebpageURL(strWebpageURL),
m_vAffiliations(vAffiliations)
{
}

SciPerson::SciPerson(std::istream& stream) : 
Indexable()
{
  m_bDeserializationSuccess = Deserialize(stream);
}

void SciPerson::SerializeValues(std::string& serializationString) const {  
  Indexable::SerializeValues(serializationString);
  AddSerialization(serializationString, m_strFirstName);
  AddSerialization(serializationString, m_strMiddleName);
  AddSerialization(serializationString, m_strLastName);
  AddSerialization(serializationString, m_strEmail);
  AddSerialization(serializationString, m_strTwitterName);
  AddSerialization(serializationString, m_strPictureURL);
  AddSerialization(serializationString, m_strWebpageURL);
  AddVectorSerialization(serializationString, m_vAffiliations);
}

bool SciPerson::DeserializeValues(std::istream& stream) {
  if (!Indexable::DeserializeValues(stream)) return false;
  if (!DeserializeValue(stream, m_strFirstName)) return false;
  if (!DeserializeValue(stream, m_strMiddleName)) return false;
  if (!DeserializeValue(stream, m_strLastName)) return false;
  if (!DeserializeValue(stream, m_strEmail)) return false;
  if (!DeserializeValue(stream, m_strTwitterName)) return false;
  if (!DeserializeValue(stream, m_strPictureURL)) return false;
  if (!DeserializeValue(stream, m_strWebpageURL)) return false;
  return DeserializeVector(stream, m_vAffiliations);
}

unsigned int SciPerson::ReplaceAffiliation(const Affiliation& aff, const Affiliation& newAff) {
    unsigned int refs = 0;
    index_t index = aff.GetIndex();
    for (size_t i = 0;i<m_vAffiliations.size();++i) {
        if (m_vAffiliations[i] == index) {
            m_vAffiliations[i] = newAff.GetIndex();
            ++refs;
        }
    }
    return refs;
}


void SciPerson::DelAffiliation(index_t index) {
    for (size_t i = 0;i<m_vAffiliations.size();++i) {
        if (m_vAffiliations[i] == index)
            m_vAffiliations.erase(m_vAffiliations.begin()+i);
    }
}

std::string SciPerson::GetDisplayName() const {
  std::string result = m_strFirstName;    
  if (!m_strMiddleName.empty()) result += std::string(" ") + m_strMiddleName;
  if (!m_strLastName.empty()) result += std::string(" ") + m_strLastName;
  return result;
}

std::string SciPerson::GetSortableDisplayName(const ConferenceDB* db) const {
  std::string result = m_strLastName;
  if (!m_strFirstName.empty()) {
      if (result.empty())
        result = m_strFirstName;
      else
        result += std::string(", ") + m_strFirstName;
  }
  if (!m_strMiddleName.empty()) {
      if (result.empty())
        result = m_strMiddleName;
      else
        result += std::string(" ") + m_strMiddleName;
  }

  std::string aff = GetDisplayAffiliations(db);

  if (!aff.empty())
      result += std::string(" - ") + aff;

  return result;
}

std::string SciPerson::GetDisplayAffiliations(const ConferenceDB* db) const {
  std::string result;
  for (size_t i = 0;i<m_vAffiliations.size();++i) {
    const Affiliation* a = db->affiliations.GetEntryByIndex(m_vAffiliations[i]);
    if (a) {
      if (result.empty()) 
        result = a->GetName();
      else {
        if (i == m_vAffiliations.size()-1)
          result += std::string(", ") + StringList::strAnd + std::string(" ") + a->GetName();
        else
          result += std::string(", ") + a->GetName();
      }
    }
  }
  return result;
}

std::vector<index_t> SciPerson::GetWorksAtThisConference(const ConferenceDB* db) const {
  std::vector<index_t> works;  
  for (size_t i = 0;i<db->works.Count();++i) {
    const SciWork* work = db->works.GetEntryByPos(i);
    const std::vector<index_t>& authors = work->GetAuthors();
    
    for (size_t j = 0;j<authors.size();++j) {
      if (m_iIndex == authors[j]) {
        works.push_back(work->GetIndex());
      }
    }
  }
  return works;
}

bool SciPerson::Validate(const ConferenceDB* confDB) const {
  for (size_t i = 0;i<m_vAffiliations.size();++i) {
    if (! confDB->affiliations.GetEntryByIndex(m_vAffiliations[i]) ) return false;
  }  
  return true;
}

unsigned int SciPerson::Replace(const SciPerson& newPers, ConferenceDB* confDB) const {
    unsigned int refs = 0;

    for (size_t i = 0;i<confDB->works.Count();++i) {
      SciWork* work = confDB->works.GetEntryByPos(i);
      refs += work->ReplaceAuthor(*this, newPers);
    }

    for (size_t i = 0;i<confDB->sessions.Count();++i) {
      Session* potentialSession = confDB->sessions.GetEntryByPos(i);
      if (potentialSession->GetSessionChair() == m_iIndex ) {
        potentialSession->SetSessionChair(newPers);
        refs++;
      }

      Panel* panel = dynamic_cast<Panel*>(potentialSession);
      if (panel) {
          refs += panel->ReplacePanelist(*this, newPers);
      }

      OrganizedSession* organized = dynamic_cast<OrganizedSession*>(potentialSession);
      if (organized) {
          refs += organized->ReplaceOrganizer(*this, newPers);
      }
    }

    for (size_t i = 0;i<confDB->talks.Count();++i) {
      Talk* potentialTalk = confDB->talks.GetEntryByPos(i);
      refs += potentialTalk->ReplaceSpeaker(*this, newPers);
    }

    return refs;
}

unsigned int SciPerson::GetReferenceCount(const ConferenceDB* confDB) const {
  unsigned int refs = (unsigned int)(GetWorksAtThisConference(confDB).size());

  for (size_t i = 0;i<confDB->sessions.Count();++i) {
    const Session* potentialSession = confDB->sessions.GetEntryByPos(i);
    if (potentialSession->GetSessionChair() == m_iIndex ) {
      refs++;
    }

    const Panel* panel = dynamic_cast<const Panel*>(potentialSession);
    if (panel) {
        for (size_t j = 0;j<panel->GetPanelists().size();++j) {
            if (panel->GetPanelists()[j] == m_iIndex)
                refs++;
        }
    }

    const OrganizedSession* organized = dynamic_cast<const OrganizedSession*>(potentialSession);
    if (organized) {
        for (size_t j = 0;j<organized->GetOrganizers().size();++j) {
            if (organized->GetOrganizers()[j] == m_iIndex)
                refs++;
        }
    }
  }

  for (size_t i = 0;i<confDB->talks.Count();++i) {
    const Talk* potentialTalk = confDB->talks.GetEntryByPos(i);

    for (size_t j = 0;j<potentialTalk->GetSpeakers().size();++j) {
        const SciPerson* potentialSpeaker = confDB->people.GetEntryByIndex(potentialTalk->GetSpeakers()[j]);
        assert(potentialSpeaker);

        if (potentialSpeaker->GetIndex() == m_iIndex ) {
          refs++;
        }
    }
  }

  return refs;
}

