#ifndef INDEXABLE_H
#define INDEXABLE_H

#include "Serializable.h"
#include <vector>
#include <cassert>

typedef unsigned int index_t;
#define INDEX_EMPTY 0

class Indexable;
class ConferenceDB;
typedef std::vector<Indexable*> IndexedVector;
typedef IndexedVector::const_iterator CIndexedVectorIter;
typedef IndexedVector::iterator IndexedVectorIter;

class IndexGenerator : public Serializable {
public:
  IndexGenerator();
  IndexGenerator(index_t iStartIndex);
  IndexGenerator(CIndexedVectorIter& iStart, CIndexedVectorIter& iEnd);  
  IndexGenerator(std::istream& stream);
  
  index_t GenIndex() {
    return ++m_iMinIndex;
  }
  
protected:
  index_t m_iMinIndex;  
  virtual void SerializeValues(std::string& serializationString) const;
  virtual bool DeserializeValues(std::istream& stream) ;
  
};

class ConferenceDB;

class Indexable : public Serializable {
public:
  Indexable() : Serializable(), m_iIndex(INDEX_EMPTY) {}
  Indexable(IndexGenerator& gen) : Serializable(), m_iIndex(gen.GenIndex()) {}
  
  virtual ~Indexable() {}
  
  void SetIndex(IndexGenerator& gen) {
    m_iIndex = gen.GenIndex();
  }
  index_t GetIndex() const {return m_iIndex;}
  
  virtual bool Validate(const ConferenceDB* confDB) const = 0;
  virtual unsigned int GetReferenceCount(const ConferenceDB* confDB) const = 0;

protected:  
  index_t m_iIndex;
  
  virtual void SerializeValues(std::string& serializationString) const {
    AddSerialization(serializationString, m_iIndex);      
  }
  
  virtual bool DeserializeValues(std::istream& stream) {
    return DeserializeValue(stream, m_iIndex);  
  }
};


template <class T, class F> class IndexTable : public Serializable {
public:
  IndexTable() : Serializable() {
  }
  
  IndexTable(std::istream& stream) : Serializable() {
    m_bDeserializationSuccess = Deserialize(stream);
  }
  
  virtual ~IndexTable() {
    for (size_t i = 0; i < db.size(); ++i) {
      delete db[i];
    }  
  }
  
  template <class CT> index_t AddEntry(CT& element) {
    element.SetIndex(gen);
    CT* pElemen = new CT(element);
    db.push_back(pElemen);
    return element.GetIndex();
  }


  template <class CT> index_t AddUniqueEntry(CT& element) {
      for (typename std::vector<T*>::iterator iter = db.begin();
           iter!=db.end();
           ++iter) {
        if (**iter == element) {
            return (*iter)->GetIndex();
        }
    }

    return AddEntry(element);
  }

  void RemoveEntryByPos(size_t position) {
    delete db[position];    
    db.erase (db.begin()+position);
  }
  
  template <class CT> void ReplaceIndexByEntry(index_t old_index, const CT& element, ConferenceDB* confDB) {
      if (INDEX_EMPTY == old_index) return;
      for (size_t i = 0; i < db.size(); ++i) {
        if (db[i]->GetIndex() == old_index) {
            db[i]->Replace(element, confDB);
            db.erase (db.begin()+i);
        }
      }
  }

  void RemoveEntryByIndex(index_t index) {
    if (INDEX_EMPTY == index) return;
    for (size_t i = 0; i < db.size(); ++i) {
      if (db[i]->GetIndex() == index) db.erase (db.begin()+i);
    }  
  }
  
  const T* GetEntryByPos(size_t pos) const {
    assert(pos < db.size());
    return db[pos];
  }
  T* GetEntryByPos(size_t pos) {
    assert(pos < db.size());
    return db[pos];
  }
  
  const T* GetEntryByIndex(index_t index) const {
    if (INDEX_EMPTY != index) {
      for (size_t i = 0; i < db.size(); ++i) {
        if (db[i]->GetIndex() == index)  return db[i];
      }  
    }
    return NULL;
  }
  T* GetEntryByIndex(index_t index) {
    if (INDEX_EMPTY != index) {
      for (size_t i = 0; i < db.size(); ++i) {
        if (db[i]->GetIndex() == index)  return db[i];
      }  
    }
    return NULL;
  }

  size_t Count() const {return db.size();}
  
  bool Validate(const ConferenceDB* confDB) const {
    for (size_t i = 0; i < db.size(); ++i) {
      if (!db[i]->Validate(confDB)) {
          return false;
      }
    }      
    return true;
  }
  
  unsigned int RemoveUnusedEntries(const ConferenceDB* confDB) {
    unsigned int iChanged = 0;
    for (size_t i = 0; i < db.size(); ++i) {
      if (db[i]->GetReferenceCount(confDB) == 0) {
          RemoveEntryByPos(i);
          i--;
          iChanged++;
      }
    }
    return iChanged;
  }

protected:
  IndexGenerator gen;
  std::vector<T*> db;
  
  virtual void SerializeValues(std::string& serializationString) const {
    gen.Serialize(serializationString);
    AddVectorSerializationWithFactory<T,F>(serializationString, db);      
  }
  
  virtual bool DeserializeValues(std::istream& stream) {
    if (!gen.Deserialize(stream)) return false;
    return DeserializeVectorWithFactory<T,F>(stream, db);
  }
  
};


#endif // INDEXABLE_H
