#ifndef BUILDING_H
#define BUILDING_H

#include <string>
#include "Serializable.h"
#include "Indexable.h"

typedef std::pair<double,double> GlobalPosition; // latitude/longitude

class Floorplan {
public:
  Floorplan(std::string desc, std::string filename, float fScale=1.0) :
  m_desc(desc),
  m_filename(filename),
  m_fScale(fScale)    
  {
  }
  
  bool operator==(const Floorplan &other) const {
    return other.m_desc == m_desc &&
           other.m_filename == m_filename &&
           other.m_fScale == m_fScale;
  }

  std::string m_desc;
  std::string m_filename;
  float m_fScale;
};

class Building : public Indexable {
public:
  Building(const std::string& strName,
           GlobalPosition pos);
  
  Building(const std::string& strName,
           GlobalPosition pos,
           const std::vector<Floorplan>& vFloorPlans);
  
  Building(std::istream& stream);
  
  const std::string& GetName() const {return m_strName;}
  void SetName(const std::string& strName)  {m_strName = strName;}
  
  const GlobalPosition& GetPos() const {return m_pos;}
  void SetPos(const GlobalPosition& pos)  {m_pos = pos;}
  
  const std::vector<Floorplan>& GetFloorplans() const {return m_vFloorPlans;}
  void SetFloorplans(const std::vector<Floorplan>& vFloorPlans)  {m_vFloorPlans = vFloorPlans;}
  void AddFloorplan(const Floorplan& floorplan)  {m_vFloorPlans.push_back(floorplan);}
  void DelFloorplan(size_t index, ConferenceDB* confDB);

  static std::string GetClassName() {return "Building";} 
  virtual std::string GetClassNameViaObject() const {return GetClassName();}   

  virtual bool Validate(const ConferenceDB*) const {return true;}
  unsigned int GetReferenceCount(const ConferenceDB* confDB) const;
  unsigned int GetFPReferenceCount(const ConferenceDB* confDB, size_t floorplanIndex) const;
  
  bool operator==(const Building &other) const {
    return other.m_strName == m_strName &&
           other.m_pos == m_pos &&
           other.m_vFloorPlans == m_vFloorPlans;
  }

protected:
  Building() : Indexable() {}
  std::string m_strName;
  GlobalPosition m_pos;
  std::vector<Floorplan> m_vFloorPlans;
  
  virtual void SerializeValues(std::string& serializationString) const;
  virtual bool DeserializeValues(std::istream& stream);
  
};


#endif // BUILDING_H
